<?php

namespace XFI\Import\Importer;

use XF\Db\Mysqli\Adapter;
use XF\Entity\User;
use XF\Import\Data\Attachment;

use XF\Import\Data\Category;
use XF\Import\Data\ConversationMaster;
use XF\Import\Data\ConversationMessage;
use XF\Import\Data\Forum;
use XF\Import\Data\LinkForum;
use XF\Import\Data\Node;
use XF\Import\Data\Poll;
use XF\Import\Data\PollResponse;
use XF\Import\Data\Post;
use XF\Import\Data\ProfilePost;
use XF\Import\Data\ProfilePostComment;
use XF\Import\Data\Reaction;
use XF\Import\Data\ReactionContent;
use XF\Import\Data\Thread;
use XF\Import\Data\UserField;
use XF\Import\Data\UserGroup;
use XF\Import\DataHelper\Moderator;
use XF\Import\DataHelper\Tag;
use XF\Import\Importer\AbstractForumImporter;
use XF\Import\PlatformUtil\Ips;
use XF\Import\StepState;
use XF\Timer;
use XF\Util\Str;

use function array_key_exists, boolval, in_array, intval, is_array;

class IpsForums extends AbstractForumImporter
{
	/**
	 * @var Adapter
	 */
	protected $sourceDb;

	public static function getListInfo()
	{
		return [
			'target' => 'XenForo',
			'source' => 'Invision Community Forums 4.x',
		];
	}

	protected function getBaseConfigDefault()
	{
		return Ips::getDefaultImportConfig();
	}

	public function validateBaseConfig(array &$baseConfig, array &$errors)
	{
		Ips::validateImportConfig($baseConfig, $errors);
		return $errors ? false : true;
	}

	public function renderBaseConfigOptions(array $vars)
	{
		$vars['db'] = Ips::getDbConfig();
		return $this->app->templater()->renderTemplate('admin:xfi_import_config_ips_forums', $vars);
	}

	protected function getStepConfigDefault()
	{
		return [
			'userGroups' => [
				'guest_group' => 2,
				'member_group' => 3,
				'admin_group' => 4,
			],
			'users' => [
				'merge_email' => false,
				'merge_name'  => false,
			],
			'polls' => [
				'which_question' => 'newest',
			],
		];
	}

	public function renderStepConfigOptions(array $vars)
	{
		$vars = array_merge_recursive(
			$this->getStepConfigDefault(),
			$vars
		);

		$config = Ips::getConfig($this->baseConfig['ips_path']);

		foreach ($vars['userGroups'] AS $name => $id)
		{
			if (!isset($config[$name]))
			{
				continue;
			}
			$vars['userGroups'][$name] = $config[$name];
		}

		return $this->app->templater()->renderTemplate('admin:xfi_import_step_config_ips_forums', $vars);
	}

	public function validateStepConfig(array $steps, array &$stepConfig, array &$errors)
	{
		if (in_array('userGroups', $steps))
		{
			if (!isset($stepConfig['userGroups']))
			{
				return false;
			}

			$sourceDb = new Adapter(
				$this->baseConfig['db'],
				$this->app->config('fullUnicode')
			);

			$groups = $sourceDb->fetchAllKeyed("
				SELECT *
				FROM core_groups
				ORDER BY g_id
			", 'g_id');

			foreach ($stepConfig['userGroups'] AS $name => $id)
			{
				if (!isset($groups[$id]))
				{
					$errors[] = \XF::phrase('xfi_user_group_id_x_not_found_in_source_database', ['id' => $id]);
					return false;
				}
			}
		}

		return true;
	}

	public function getSteps()
	{
		return [
			'userGroups' => [
				'title' => \XF::phrase('user_groups'),
			],
			'userFields' => [
				'title' => \XF::phrase('custom_user_fields'),
			],
			'users' => [
				'title' => \XF::phrase('users'),
				'depends' => ['userGroups', 'userFields'],
			],
			//			'ignoredUsers' => [
			//				'title' => \XF::phrase('xfi_ignored_users'),
			//				'depends' => ['users']
			//			],
			'privateMessages' => [
				'title' => \XF::phrase('xfi_import_private_messages'),
				'depends' => ['users'],
			],
			'statusUpdates' => [
				'title' => \XF::phrase('profile_posts'),
				'depends' => ['users'],
			],
			'forums' => [
				'title' => \XF::phrase('forums'),
				'depends' => ['userGroups'],
			],
			'moderators' => [
				'title' => \XF::phrase('moderators'),
				'depends' => ['forums', 'users'],
			],
			'threads' => [
				'title' => \XF::phrase('threads'),
				'depends' => ['forums', 'users'],
			],
			'posts' => [
				'title' => \XF::phrase('posts'),
				'depends' => ['threads'],
			],
			'archivePosts' => [
				'title' => \XF::phrase('xfi_archived_posts'),
				'depends' => ['posts', 'threads'],
			],
			'contentTags' => [
				'title' => \XF::phrase('tags'),
				'depends' => ['posts'],
			],
			'polls' => [
				'title' => \XF::phrase('polls'),
				'depends' => ['posts'],
			],
			'attachments' => [
				'title' => \XF::phrase('attachments'),
				'depends' => ['posts'],
			],
			'reactions' => [
				'title' => \XF::phrase('reaction_definitions'),
			],
			'reactionContent' => [
				'title' => \XF::phrase('reaction_content'),
				'depends' => ['posts', 'reactions'],
			],
		];
	}

	protected function doInitializeSource()
	{
		$this->sourceDb = new Adapter(
			$this->baseConfig['db'],
			$this->app->config('fullUnicode')
		);
	}

	// ############################## STEP: USER GROUPS #########################

	protected $userGroups;

	protected function getUserGroups()
	{
		if ($this->userGroups === null)
		{
			$this->userGroups = $this->sourceDb->fetchAllKeyed("
				SELECT ugroup.*, lang.word_custom AS g_title
				FROM core_groups AS ugroup
				LEFT JOIN core_sys_lang_words AS lang ON
					(lang.lang_id = 1 AND lang.word_key = CONCAT('core_group_', ugroup.g_id))
				LEFT JOIN core_moderators AS moderator ON
					(ugroup.g_id = moderator.id AND moderator.type = 'g')
				ORDER BY ugroup.g_id
			", 'g_id');
		}
		return $this->userGroups;
	}

	protected function calculateGroupPerms(array $group)
	{
		$p = [];

		$this->calculateGeneralPerms($p, $group);
		$this->calculateForumPerms($p, $group);
		$this->calculateForumModPerms($p, $group);
		$this->calculateAvatarPerms($p, $group);

		return $p;
	}

	protected function calculateGeneralPerms(array &$p, array $group)
	{
		if ($group['g_view_board'])
		{
			$p['general']['view'] = 'allow';
			$p['general']['viewNode'] = 'allow';
			$p['forum']['viewAttachment'] = 'allow';
			$p['forum']['viewContent'] = 'allow';
			$p['forum']['viewOthers'] = 'allow';
		}

		if ($group['g_mem_info'])
		{
			$p['general']['viewProfile'] = 'allow';
			$p['general']['viewMemberList'] = 'allow';
			$p['profilePost']['view'] = 'allow';
			$p['profilePost']['post'] = 'allow';
			$p['profilePost']['comment'] = 'allow';
		}

		if ($group['g_avoid_flood'])
		{
			$p['general']['bypassFloodCheck'] = 'allow';
		}

		if ($group['g_use_search'])
		{
			$p['general']['search'] = 'allow';
		}

		// this is mapped from max number of +ve reputation points awardable in 24h
		if ($group['g_rep_max_positive'])
		{
			$p['forum']['react'] = 'allow';
			$p['profilePost']['react'] = 'allow';
		}

		if ($group['g_use_pm'])
		{
			$p['conversation']['start'] = 'allow';
			$p['conversation']['receive'] = 'allow';
			$p['conversation']['maxRecipients'] = $group['g_max_mass_pm']; // should be max 500
		}
	}

	protected function calculateForumPerms(array &$p, array $group)
	{
		if (isset($group['g_delete_own_posts']) || $group['g_bitoptions'] & 128) // gbw_soft_delete_own
		{
			$p['forum']['deleteOwnPost'] = 'allow';
		}

		if (isset($group['g_delete_own_topics']) || $group['g_bitoptions'] & 256) // gbw_soft_delete_own_topic
		{
			$p['forum']['deleteOwnThread'] = 'allow';
		}

		if (isset($group['g_edit_posts']))
		{
			$p['forum']['editOwnPost'] = 'allow';
		}

		if (isset($group['g_edit_cutoff']))
		{
			$p['forum']['editOwnPostTimeLimit'] = $group['g_edit_cutoff'];
		}

		if (isset($group['g_attach_max']) && intval($group['g_attach_max']) >= 0)
		{
			$p['forum']['uploadAttachment'] = 'allow';
		}

		if (isset($group['g_vote_polls']))
		{
			$p['forum']['votePoll'] = 'allow';
		}
	}

	protected function calculateForumModPerms(array &$p, array $group)
	{
		if (isset($group['perms']))
		{
			if ($group['perms'] == '*') // all current and future perms
			{
				$p['forum']['lockUnlockThread'] = 'allow';
				$p['forum']['viewDeleted'] = 'allow';
				$p['forum']['stickUnstickThread'] = 'allow';
				$p['forum']['manageAnyThread'] = 'allow';
				$p['forum']['react'] = 'allow';
				$p['profilePost']['react'] = 'allow';
				$p['conversation']['start'] = 'allow';
				$p['conversation']['receive'] = 'allow';
				$p['conversation']['maxRecipients'] = $group['g_max_mass_pm']; // should be max 500
			}
			else if ($modPerms = $this->decodeValue($group['perms'], 'json-array'))
			{
				if ($modPerms['can_lock_content']
					|| $modPerms['can_unlock_content']
					|| $modPerms['can_lock_topic']
					|| $modPerms['can_unlock_topic']
				)
				{
					$p['forum']['lockUnlockThread'] = 'allow';
				}

				if ($modPerms['can_view_hidden_content']
					|| $modPerms['can_view_hidden_topic']
					|| $modPerms['can_view_hidden_post']
				)
				{
					$p['forum']['viewDeleted'] = 'allow';
				}

				if ($modPerms['can_pin_content']
					|| $modPerms['can_unpin_content']
					|| $modPerms['can_pin_topic']
					|| $modPerms['can_unpin_topic']
				)
				{
					$p['forum']['stickUnstickThread'] = 'allow';
				}

				if (($modPerms['can_move_content'] || $modPerms['can_move_topic'])
					&& ($modPerms['can_split_merge_content'] || $modPerms['can_split_merge_topic'])
					&& ($modPerms['can_edit_content'] || $modPerms['can_edit_topic'])
				)
				{
					$p['forum']['manageAnyThread'] = 'allow';
				}
			}
		}
	}

	protected function calculateAvatarPerms(array &$p, array $group)
	{
		$max = intval($group['g_photo_max_vars']); // take the first value from '500:170:240'
		if ($max)
		{
			$p['avatar']['allowed'] = 'allow';
			if ($max > 2147483647)
			{
				$p['avatar']['maxFileSize'] = -1;
			}
			else
			{
				$p['avatar']['maxFileSize'] = $max;
			}
		}
	}

	public function stepUserGroups(StepState $state, array $stepConfig)
	{
		$groups = $this->getUserGroups();

		$this->session->extra['groupConfig'] = $stepConfig;

		foreach ($groups AS $oldId => $group)
		{
			switch ($oldId)
			{
				case $stepConfig['guest_group']: // guests
					$this->logHandler(UserGroup::class, $oldId, User::GROUP_GUEST);
					break;

				case $stepConfig['member_group']: // registered
					$this->logHandler(UserGroup::class, $oldId, User::GROUP_REG);
					break;

				case $stepConfig['admin_group']: // admins
					$this->logHandler(UserGroup::class, $oldId, User::GROUP_ADMIN);
					break;

				case 6: // mods (not configurable but this group is created by default)
					$this->logHandler(UserGroup::class, $oldId, User::GROUP_MOD);
					break;

				default:

					$data = [
						'title' => strip_tags($group['g_title']), // don't allow HTML for titles
						'user_title' => $group['g_title'],
						'display_style_priority' => 5,
					];

					/** @var UserGroup $import */
					$import = $this->newHandler(UserGroup::class);
					$import->bulkSet($data);
					$import->setPermissions(
						$this->calculateGroupPerms($group)
					);
					$import->save($oldId);
			}

			$state->imported++;
		}

		return $state->complete();
	}

	// ############################## STEP: USER FIELDS #########################

	public function stepUserFields(StepState $state)
	{
		$fields = $this->sourceDb->fetchAllKeyed("
			SELECT pfields_data.*, pfields_groups.*,
				lang_title.word_custom AS pf_title,
				lang_desc.word_custom AS pf_desc
			FROM core_pfields_data AS pfields_data
			INNER JOIN core_pfields_groups AS pfields_groups ON
				(pfields_groups.pf_group_id = pfields_data.pf_group_id)
			LEFT JOIN core_sys_lang_words AS lang_title ON
				(lang_title.lang_id = 1 AND lang_title.word_key = CONCAT('core_pfield_', pfields_data.pf_id))
			LEFT JOIN core_sys_lang_words AS lang_desc ON
				(lang_desc.lang_id = 1 AND lang_desc.word_key = CONCAT('core_pfield_', pfields_data.pf_id, '_desc'))
		", 'pf_id');

		$existingFields = $this->db()->fetchPairs("SELECT field_id, field_id FROM xf_user_field");

		foreach ($fields AS $oldId => $field)
		{
			$fieldId = $this->convertToUniqueId($field['pf_title'], $existingFields, 25);

			switch ($fieldId)
			{
				case 'facebook':
				case 'skype':
				case 'twitter':
				case 'gender':
				case 'website_url':
				case 'location':
				case 'interests':
				case 'about_me':
					// just store the mapping, no need to import these
					$this->logHandler(UserField::class, $oldId, $fieldId);
					break;

				default:
					$oldFieldType = $field['pf_type'];
					$newFieldType = Ips::getFieldType($oldFieldType, $field['pf_multiple']);

					if (!$newFieldType)
					{
						break;
					}

					if ($field['pf_admin_only'])
					{
						$field['pf_member_hide'] = true;
						$field['pf_member_edit'] = false;
					}

					$displayGroup = 'personal';
					$userEditable = ($field['pf_member_edit'] ? 'yes' : 'never');
					$viewableProfile = (!$field['pf_member_hide']);

					$data = $this->mapXfKeys($field, [
						'display_order' => 'pf_position',
						'max_length' => 'pf_max_input',
						'required' => 'pf_not_null',
						'show_registration' => 'pf_show_on_reg',
					]);
					$data['field_id'] = $fieldId;
					$data['field_type'] = $newFieldType;
					$data['display_group'] = $displayGroup;
					$data['user_editable'] = $userEditable;
					$data['viewable_profile'] = $viewableProfile;

					$data['match_params'] = [];
					if ($field['pf_input_format'])
					{
						$data['match_type'] = 'regex';
						$data['match_params']['regex'] = Ips::convertFieldMatchTypeToRegex($field['pf_input_format']);
					}
					else
					{
						$data['match_type'] = Ips::getMatchType($oldFieldType);
					}

					if (Ips::isFieldChoiceType($oldFieldType))
					{
						$data['field_choices'] = Ips::convertFieldChoices($oldFieldType, $field['pf_content']);
					}

					/** @var UserField $import */
					$import = $this->newHandler(UserField::class);
					$import->setTitle($field['pf_title'], $field['pf_desc']);
					$import->bulkSet($data);

					$import->save($oldId);

					$state->imported++;
			}
		}

		return $state->complete();
	}

	// ############################## STEP: USERS #############################

	/**
	 * Checks that the $permissions array given has the admin permission specified
	 *
	 * @param array  $adminPermissions
	 * @param string $appName
	 * @param string $moduleName
	 * @param string $permName
	 *
	 * @return boolean
	 */
	protected function hasAdminPermission(array $adminPermissions, $appName, $moduleName = null, $permName = null)
	{
		if (!$adminPermissions)
		{
			return false;
		}

		if (!in_array($appName, $adminPermissions['applications']))
		{
			return false;
		}

		if (isset($moduleName))
		{
			if (!in_array($moduleName, $adminPermissions['modules']))
			{
				return false;
			}

			if (!array_key_exists($moduleName, $adminPermissions['items'][$appName]))
			{
				return false;
			}

			if (isset($permName) && !in_array($permName, $adminPermissions['items'][$appName][$moduleName]))
			{
				return false;
			}

			return true;
		}

		return true;
	}

	protected function isUserAdmin(array $user)
	{
		if ($user['user_row_perm_cache'] || $user['group_row_perm_cache'])
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	protected function getAdminPermissions(array $user)
	{
		$permCache = null;

		if ($user['user_row_perm_cache'] == '*')
		{
			$permCache = '*';
		}
		else if ($user['group_row_perm_cache'] == '*')
		{
			$permCache = '*';
		}

		if ($permCache)
		{
			return $permCache;
		}

		if ($user['user_row_perm_cache'])
		{
			$permCache = $user['user_row_perm_cache'];
		}
		else
		{
			$permCache = $user['group_row_perm_cache'];
		}

		return $this->decodeValue($permCache, 'json-array');
	}

	protected function isUserSuperMod(array $user)
	{
		if (isset($user['perms'])) // user is a moderator
		{
			if ($user['perms'] == '*')
			{
				// user has all permissions on all forums
				return true;
			}
			else if ($modPerms = $this->decodeValue($user['perms'], 'json-array'))
			{
				// user has selective permissions on all forums
				return (isset($modPerms['forums']) && $modPerms['forums'] == -1);
			}
		}

		foreach ($this->getGroupsForUser($user) AS $group)
		{
			if (isset($group['perms'])) // group is a moderator group
			{
				if ($group['perms'] == '*')
				{
					// group has all permissions on all forums
					return true;
				}
				else if ($modPerms = $this->decodeValue($group['perms'], 'json-array'))
				{
					// group has selective permissions on all forums
					return (isset($modPerms['forums']) && $modPerms['forums'] == -1);
				}
			}
		}

		return false;
	}

	/**
	 * Fetches an array of all user groups to which the user belongs
	 *
	 * @param array $user
	 *
	 * @return array
	 */
	protected function getGroupsForUser(array $user)
	{
		$groupCache = $this->getUserGroups();

		$groups = [
			$user['member_group_id'] => $groupCache[$user['member_group_id']],
		];

		if ($user['mgroup_others'])
		{
			foreach ($this->decodeValue($user['mgroup_others'], 'list-comma-ips') AS $groupId)
			{
				if (isset($groupCache[$groupId]))
				{
					$groups[$groupId] = $groupCache[$groupId];
				}
			}
		}

		return $groups;
	}

	public function getStepEndUsers()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(member_id) FROM core_members") ?: 0;
	}

	protected $adminPermissions;

	public function stepUsers(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$users = $this->sourceDb->fetchAllKeyed("
			SELECT
				moderator.*,
			    pfields_content.*,
				uapr.row_perm_cache AS user_row_perm_cache,
			    gapr.row_perm_cache AS group_row_perm_cache,
			    members.*
			FROM core_members AS members
			LEFT JOIN core_pfields_content AS pfields_content ON
				(pfields_content.member_id = members.member_id)
			LEFT JOIN core_admin_permission_rows AS uapr ON
				(uapr.row_id = members.member_id AND uapr.row_id_type = 'member')
			LEFT JOIN core_admin_permission_rows AS gapr ON
				(gapr.row_id = members.member_group_id AND gapr.row_id_type = 'group')
			LEFT JOIN core_moderators AS moderator ON
				(members.member_id = moderator.id AND moderator.type = 'm')
			WHERE members.member_id > ? AND members.member_id <= ?
			ORDER BY members.member_id
			LIMIT {$limit}
		", 'member_id', [$state->startAfter, $state->end]);

		if (!$users)
		{
			return $state->complete();
		}

		foreach ($users AS $oldId => $user)
		{
			$state->startAfter = $oldId;

			$import = $this->setupImportUser($user);
			if ($this->importUser($oldId, $import, $stepConfig))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected $groupMap;

	protected $fieldMap;

	protected $oldFields;

	protected function setupImportUser(array $user)
	{
		/** @var \XF\Import\Data\User $import */
		$import = $this->newHandler(\XF\Import\Data\User::class);

		if ($user['name'] === '')
		{
			$user['name'] = 'Member ' . $user['member_id'];
		}

		$userData = $this->mapXfKeys($user, [
			'username' => 'name',
			'email',
			'last_activity',
			'register_date' => 'joined',
			'message_count' => 'member_posts',
		]);

		$import->bulkSetDirect('user', $userData);

		// groups
		$groupConfig = $this->session->extra['groupConfig'];

		// handle degenerate user group info
		if (empty($user['member_group_id']) || !isset($this->typeMap('user_group')[$user['member_group_id']]))
		{
			$user['member_group_id'] = $groupConfig['member_group'];
		}

		// user groups
		$import->user_group_id = $this->lookupId('user_group', $user['member_group_id'], User::GROUP_REG);
		if ($user['mgroup_others'])
		{
			$import->secondary_group_ids = $this->lookup('user_group', $this->decodeValue($user['mgroup_others'], 'list-comma-ips'));
		}

		// banned state
		$import->is_banned = ($user['temp_ban'] <> 0);
		if ($import->is_banned)
		{
			if ($user['temp_ban'] >= 1)
			{
				// temp ban until timestamp
				$endDate = $user['temp_ban'];
			}
			else
			{
				// permanent ban
				$endDate = 0;
			}

			$import->setBan([
				'ban_user_id' => 0,
				'ban_date' => 0,
				'end_date' => $endDate,
			]);
		}

		// user state
		if ($user['members_bitoptions'] & 1073741824)
		{
			$import->user_state = 'email_confirm';
		}
		else
		{
			$import->user_state = 'valid';
		}

		// is admin
		$import->is_admin = $this->isUserAdmin($user);
		if ($import->is_admin)
		{
			$adminPermissions = $this->getAdminPermissions($user);

			if ($adminPermissions == '*')
			{
				if ($this->adminPermissions === null)
				{
					$this->adminPermissions = $this->db()->fetchAllColumn("
						SELECT admin_permission_id
						FROM xf_admin_permission
						ORDER BY display_order
					");
				}
				$importAdminPerms = $this->adminPermissions;
			}
			else
			{
				$importAdminPerms = [];

				if ($this->hasAdminPermission($adminPermissions, 'core', 'applications'))
				{
					$importAdminPerms[] = 'option';
					$importAdminPerms[] = 'import';
					$importAdminPerms[] = 'upgradeXenForo';
					$importAdminPerms[] = 'addOn';
					$importAdminPerms[] = 'advertising';
					$importAdminPerms[] = 'attachment';
					$importAdminPerms[] = 'payment';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'editor'))
				{
					$importAdminPerms[] = 'bbCodeSmilie';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'settings', 'advanced_manage_tasks'))
				{
					$importAdminPerms[] = 'cron';
					$importAdminPerms[] = 'viewStatistics';
					$importAdminPerms[] = 'viewLogs';
					$importAdminPerms[] = 'rebuildCache';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'customization'))
				{
					$importAdminPerms[] = 'style';
					$importAdminPerms[] = 'language';
					$importAdminPerms[] = 'notice';
					$importAdminPerms[] = 'widget';
					$importAdminPerms[] = 'help';
					$importAdminPerms[] = 'reaction';
					$importAdminPerms[] = 'navigation';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'languages'))
				{
					$importAdminPerms[] = 'language';
				}

				if ($this->hasAdminPermission($adminPermissions, 'forums', 'forums', 'forums_manage'))
				{
					$importAdminPerms[] = 'node';
					$importAdminPerms[] = 'tags';
					$importAdminPerms[] = 'thread';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'members'))
				{
					$importAdminPerms[] = 'user';
					$importAdminPerms[] = 'trophy';
					$importAdminPerms[] = 'userUpgrade';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'members', 'member_ban'))
				{
					$importAdminPerms[] = 'ban';
					$importAdminPerms[] = 'warning';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'membersettings', 'profilefields_manage'))
				{
					$importAdminPerms[] = 'userField';
				}

				if ($this->hasAdminPermission($adminPermissions, 'core', 'members', 'groups_manage'))
				{
					$importAdminPerms[] = 'userGroup';
				}
			}

			$adminData = [
				'permission_cache' => $importAdminPerms,
			];
			$import->setAdmin($adminData);
		}

		// default watch state
		$import->creation_watch_state = '';
		$import->interaction_watch_state = '';

		$autoTrack = $this->decodeValue($user['auto_track'], 'json-array');
		if ($autoTrack)
		{
			if ($autoTrack['comments'] || $autoTrack['content'])
			{
				if ($autoTrack['method'] == 'immediate')
				{
					$import->creation_watch_state = 'watch_no_email';
					$import->interaction_watch_state = 'watch_no_email';
				}
				else
				{
					$import->creation_watch_state = 'watch_email';
					$import->interaction_watch_state = 'watch_email';
				}
			}
		}

		// authentication
		if ($user['members_pass_salt'] && Str::strlen($user['members_pass_salt']) === 32)
		{
			$scheme = 'XF:IpsForums3x';
		}
		else
		{
			$scheme = 'XF:IpsForums4x';
		}
		$import->setPasswordData($scheme, [
			'hash' => $user['members_pass_hash'],
			'salt' => $user['members_pass_salt'],
		]);

		$import->setRegistrationIp($user['ip_address']);

		$import->timezone = $user['timezone'] == 'UTC' ? 'Europe/London' : $user['timezone'];

		if ($user['pp_photo_type'] == 'gravatar')
		{
			$import->gravatar = $user['pp_gravatar'];
		}
		else if ($user['pp_photo_type'] == 'custom' && $user['pp_main_photo'])
		{
			$avatarPath = "{$this->baseConfig['ips_path']}/uploads/$user[pp_main_photo]";
			if (file_exists($avatarPath) && is_readable($avatarPath))
			{
				$import->setAvatarPath($avatarPath);
			}
		}

		// custom title
		if ($user['member_title'])
		{
			$import->custom_title = $user['member_title'];
		}

		// custom user fields
		if ($this->oldFields === null)
		{
			$this->oldFields = $this->sourceDb->fetchAllKeyed("
				SELECT *
				FROM core_pfields_data
				ORDER BY pf_id
			", 'pf_id');
		}

		$fieldValues = [];

		foreach ($this->typeMap('user_field') AS $oldId => $newId)
		{
			if (isset($user["field_$oldId"]) && $user["field_$oldId"] !== '' && isset($this->oldFields[$oldId]))
			{
				$fieldValue = $user["field_$oldId"];

				switch ($newId)
				{
					// map these custom fields to our hard-coded fields
					case 'website':
						$import->website = $fieldValue;
						break;

					case 'location':
						$import->location = $fieldValue;
						break;

					case 'interests':
						$import->about .= "\n\n" . $fieldValue;
						break;

					default:
						// handle IPS custom fields that we also treat as custom
						$oldField = $this->oldFields[$oldId];
						$fieldChoices = @json_decode($oldField['pf_content'], true) ?: [];

						$fieldValues[$newId] = Ips::getFieldValue(
							$oldField['pf_type'],
							$fieldValue,
							$oldField['pf_multiple'],
							$fieldChoices,
							$this->sourceDb
						);
				}
			}
		}

		if ($fieldValues)
		{
			$import->setCustomFields($fieldValues);
		}

		// user profile
		$profileData = $this->mapXfKeys($user, [
			'dob_day' => 'bday_day',
			'dob_month' => 'bday_month',
			'dob_year' => 'bday_year',
		]);
		$import->bulkSetDirect('profile', array_map('intval', $profileData));

		$import->signature = Ips::convertContentToBbCode($user['signature']);
		$import->receive_admin_email = $user['allow_admin_mails'];
		$import->content_show_signature = $user['members_bitoptions'] & 65536 ? true : false;
		$import->allow_send_personal_conversation = ($user['members_disable_pm'] ? 'none' : 'members');
		$import->allow_post_profile = ($user['pp_setting_count_comments'] ? 'members' : 'none');
		$import->show_dob_year = true;
		$import->show_dob_date = true;

		return $import;
	}

	// ########################### STEP: PRIVATE MESSAGES ###############################

	public function getStepEndPrivateMessages()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(mt_id)
			FROM core_message_topics
			WHERE mt_is_draft = 0
				AND mt_is_deleted = 0
				AND mt_is_system = 0
		") ?: 0;
	}

	public function stepPrivateMessages(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 300;
		$timer = new Timer($maxTime);

		$topics = $this->sourceDb->fetchAllKeyed("
			SELECT mtopics.*, members.name AS mt_starter_name
			FROM core_message_topics AS mtopics
			INNER JOIN core_members AS members ON
				(mtopics.mt_starter_id = members.member_id)
			WHERE mtopics.mt_id > ? AND mtopics.mt_id <= ?
			AND mt_is_draft = 0
			AND mt_is_deleted = 0
			AND mt_is_system = 0
			ORDER BY mtopics.mt_id
			LIMIT {$limit}
		", 'mt_id', [$state->startAfter, $state->end]);

		if (!$topics)
		{
			return $state->complete();
		}

		foreach ($topics AS $oldId => $topic)
		{
			$state->startAfter = $oldId;

			$userMap = $this->sourceDb->fetchAll("
				SELECT topicUserMap.*,
					members.name AS map_user_name
				FROM core_message_topic_user_map AS topicUserMap
				INNER JOIN core_members AS members ON
					(topicUserMap.map_user_id = members.member_id)
				WHERE topicUserMap.map_topic_id = {$this->sourceDb->quote($topic['mt_id'])}
				ORDER BY topicUserMap.map_id
			");

			$this->lookup('user', array_column($userMap, 'map_user_id'));

			/** @var ConversationMaster $import */
			$import = $this->newHandler(ConversationMaster::class);

			foreach ($userMap AS $user)
			{
				$targetUserId = $this->lookupId('user', $user['map_user_id']);
				if (!$targetUserId)
				{
					continue;
				}

				if ($user['map_user_active'] == 0)
				{
					$recipientState = 'deleted_ignored';
				}
				else
				{
					$recipientState = 'active';
				}

				$import->addRecipient($targetUserId, $recipientState, [
					'last_read_date' => $user['map_read_time'],
					'recipient_state' => $recipientState,
					'is_unread' => $user['map_has_unread'],
					'starred' => 0,
				]);
			}

			$conversation = [
				'title' => $topic['mt_title'],
				'user_id' => $this->lookupId('user', $topic['mt_starter_id']),
				'username' => $topic['mt_starter_name'],
				'start_date' => $topic['mt_date'],
				'open_invite' => 0,
				'conversation_open' => 1,
			];

			$posts = $this->sourceDb->fetchAll("
				SELECT messagePosts.*,
					members.name AS msg_author_name
				FROM core_message_posts AS messagePosts
				INNER JOIN core_members AS members ON
					(messagePosts.msg_author_id = members.member_id)
				WHERE messagePosts.msg_topic_id = {$topic['mt_id']}
				ORDER BY messagePosts.msg_date
			");

			foreach ($posts AS $post)
			{
				$message = Ips::convertContentToBbCode($post['msg_post'], 'convMessage');

				/** @var ConversationMessage $importMessage */
				$importMessage = $this->newHandler(ConversationMessage::class);
				$importMessage->bulkSet([
					'message_date' => $post['msg_date'],
					'user_id' => $this->lookupId('user', $post['msg_author_id'], 0),
					'username' => $post['msg_author_name'],
				]);

				$helper = $this->getHelper();

				$quotedMessageIds = $helper->getQuotedContentIds($message, 'convMessage');
				$mentionedUserIds = $helper->getMentionedUserIds($message);

				$this->lookup('conversation_message', $quotedMessageIds);
				$this->lookup('user', $mentionedUserIds);

				$message = $helper->rewriteQuotesInBbCode($message, 'convMessage', 'conversation_message');
				$message = $helper->rewriteMentionsInBbCode($message);
				$importMessage->message = $message;

				$import->addMessage($post['msg_id'], $importMessage);
			}

			$import->bulkSet($conversation);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: STATUS UPDATES ###############################

	public function getStepEndStatusUpdates()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(status_id) FROM core_member_status_updates") ?: 0;
	}

	public function stepStatusUpdates(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$statusUpdates = $this->sourceDb->fetchAllKeyed("
			SELECT msus.*,
				members.name AS status_member_name
			FROM core_member_status_updates AS msus
			INNER JOIN core_members AS members ON
				(msus.status_member_id = members.member_id)
			WHERE msus.status_id > ? AND msus.status_id <= ? AND msus.status_approved = 1
			ORDER BY msus.status_id
			LIMIT {$limit}
		", 'status_id', [$state->startAfter, $state->end]);

		if (!$statusUpdates)
		{
			return $state->complete();
		}

		$mapUserIds = [];

		foreach ($statusUpdates AS $statusUpdate)
		{
			$mapUserIds[] = $statusUpdate['status_member_id'];
			$mapUserIds[] = $statusUpdate['status_author_id'];
		}

		$this->lookup('user', $mapUserIds);

		foreach ($statusUpdates AS $oldId => $statusUpdate)
		{
			$state->startAfter = $oldId;

			$profileUserId = $this->lookupId('user', $statusUpdate['status_member_id']);
			$userId = $this->lookupId('user', $statusUpdate['status_author_id']);
			$username = $statusUpdate['status_member_name'];

			if (!$profileUserId)
			{
				continue;
			}

			/** @var ProfilePost $import */
			$import = $this->newHandler(ProfilePost::class);

			if ($statusUpdate['status_replies'])
			{
				$replies = $this->sourceDb->fetchAllKeyed("
					SELECT replies.*, members.name
					FROM core_member_status_replies AS replies
					INNER JOIN core_members AS members ON
						(replies.reply_member_id = members.member_id)
					WHERE replies.reply_status_id = ? AND replies.reply_approved = 1
					ORDER BY replies.reply_date
				", 'reply_id', $statusUpdate['status_id']);

				if ($replies)
				{
					$this->lookup('user', array_column($replies, 'reply_member_id'));

					foreach ($replies AS $oldCommentId => $reply)
					{
						/** @var ProfilePostComment $comment */
						$comment = $this->newHandler(ProfilePostComment::class);
						$comment->user_id = $this->lookupId('user', $reply['reply_member_id']);
						$comment->username = $reply['name'];
						$comment->comment_date = $reply['reply_date'];
						$comment->message = Ips::convertContentToBbCode($reply['reply_content']);

						$comment->setLoggedIp($reply['reply_ip_address']);

						$import->addComment($oldCommentId, $comment);
					}
				}
			}

			$import->profile_user_id = $profileUserId;
			$import->user_id = $userId;
			$import->username = $username;
			$import->post_date = $statusUpdate['status_date'];
			$import->message = Ips::convertContentToBbCode($statusUpdate['status_content']);
			$import->message_state = 'visible';
			$import->comment_count = $statusUpdate['status_replies'];

			$import->setLoggedIp($statusUpdate['status_author_ip']);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: FORUMS ###############################

	public function stepForums(StepState $state)
	{
		$forums = $this->sourceDb->fetchAllKeyed("
			SELECT forum.*, lang_title.word_custom AS title, lang_desc.word_custom AS description
			FROM forums_forums AS forum
			LEFT JOIN core_sys_lang_words AS lang_title ON
				(lang_title.lang_id = 1 AND lang_title.word_key = CONCAT('forums_forum_', forum.id))
			LEFT JOIN core_sys_lang_words AS lang_desc ON
				(lang_desc.lang_id = 1 AND lang_desc.word_key = CONCAT('forums_forum_', forum.id, '_desc'))
			ORDER BY forum.id
		", 'id');

		if (!$forums)
		{
			return $state->complete();
		}

		$nodeTree = [];
		foreach ($forums AS $nodeId => $forum)
		{
			$nodeTree[$forum['parent_id']][] = $forum;
		}

		$state->imported = $this->importNodeTree($forums, $nodeTree);

		return $state->complete();
	}

	protected function importNodeTree(array $nodes, array $tree, $oldParentId = -1, $newParentId = 0)
	{
		if (!isset($tree[$oldParentId]))
		{
			return 0;
		}

		$total = 0;

		foreach ($tree[$oldParentId] AS $node)
		{
			$oldNodeId = $node['id'];

			/** @var Node $importNode */
			$importNode = $this->newHandler(Node::class);
			$importNode->bulkSet($this->mapXfKeys($node, [
				'title',
				'description',
				'display_order' => 'position',
			]));
			$importNode->parent_node_id = $newParentId;

			if ($node['redirect_on'] && $node['redirect_url'])
			{
				$nodeTypeId = 'LinkForum';

				/** @var LinkForum $importType */
				$importType = $this->newHandler(LinkForum::class);
				$importType->link_url = $node['redirect_url'];
			}
			else if ($node['sub_can_post']) // forum
			{
				$nodeTypeId = 'Forum';

				/** @var Forum $importType */
				$importType = $this->newHandler(Forum::class);
				$importType->discussion_count = $node['topics'];
				$importType->message_count = $node['posts'] + $node['topics'];
				$importType->last_post_date = $node['last_post'];
				$importType->last_post_username = $node['last_poster_name'];
			}
			else
			{
				$nodeTypeId = 'Category';

				/** @var Category $importType */
				$importType = $this->newHandler(Category::class);
			}

			$importNode->setType($nodeTypeId, $importType);

			$newNodeId = $importNode->save($oldNodeId);
			if ($newNodeId)
			{
				$total++;
				$total += $this->importNodeTree($nodes, $tree, $oldNodeId, $newNodeId);
			}
		}

		return $total;
	}

	// ########################### STEP: MODERATORS ###############################

	public function stepModerators(StepState $state)
	{
		$moderators = $this->sourceDb->fetchAll("
			SELECT moderators.*, members.member_id
			FROM core_moderators AS moderators
			INNER JOIN core_members AS members ON
				(moderators.id = members.member_id AND moderators.type = 'm')
		");

		$moderatorsMap = [];

		foreach ($moderators AS $moderator)
		{
			$moderatorsMap[$moderator['member_id']] = $moderator;
		}

		if (!$moderatorsMap)
		{
			return $state->complete();
		}

		$this->typeMap('node');
		$this->lookup('user', array_keys($moderatorsMap));

		/** @var Moderator $modHelper */
		$modHelper = $this->getDataHelper(Moderator::class);

		foreach ($moderatorsMap AS $oldUserId => $moderator)
		{
			$newUserId = $this->lookupId('user', $oldUserId);

			if (!$newUserId)
			{
				continue;
			}

			if ($this->isUserSuperMod($moderator))
			{
				// note this approach is slightly different but arguably more correct
				// in XF-land by default super moderators are just in the "Moderating"
				// group, and by default this group has all of the permissions a super
				// moderator should have so this should work better overall.
				$modHelper->importModerator(
					$newUserId,
					true,
					[User::GROUP_MOD],
					[]
				);

				$state->imported++;
			}
			else
			{
				$forumIds = [];
				$perms = json_decode($moderator['perms'], true);
				if (isset($perms['forums']))
				{
					$forumIds = $perms['forums'];
				}

				if ($forumIds && is_array($forumIds))
				{
					$forumPerms = $this->calculateForumModeratorPermissions($moderator);

					foreach ($forumIds AS $forumId)
					{
						$newNodeId = $this->lookupId('node', $forumId);
						if (!$newNodeId)
						{
							continue;
						}

						$modHelper->importContentModerator($newUserId, 'node', $newNodeId, $forumPerms);

						$state->imported++;
					}
				}
			}
		}

		return $state->complete();
	}

	protected function calculateForumModeratorPermissions(array $moderator)
	{
		$modPerms = json_decode($moderator['perms'], true);

		$general = [];

		if (!empty($modPerms['can_use_ip_tools']))
		{
			$general['viewIps'] = true;
		}

		if (!empty($modPerms['can_flag_as_spammer']))
		{
			$general['cleanSpam'] = true;
		}

		$forum = [
			'viewModerated' => true,
			'approveUnapprove' => true,
		];

		if (!empty($modPerms['can_edit_content']) || !empty($modPerms['can_edit_post']))
		{
			$forum['editAnyPost'] = true;
		}

		if (!empty($modPerms['can_edit_content']) || !empty($modPerms['can_edit_topic']))
		{
			$forum['manageAnyThread'] = true;
		}

		if (!empty($modPerms['can_pin_content'])
			|| !empty($modPerms['can_unpin_content'])
			|| !empty($modPerms['can_pin_topic'])
			|| !empty($modPerms['can_unpin_topic']))
		{
			$forum['stickUnstickThread'] = true;
		}

		if (!empty($modPerms['can_lock_content'])
			|| !empty($modPerms['can_unlock_content'])
			|| !empty($modPerms['can_lock_topic'])
			|| !empty($modPerms['can_unlock_topic'])
		)
		{
			$forum['lockUnlockThread'] = true;
		}

		if (!empty($modPerms['can_hide_content']) || !empty($modPerms['can_hide_post']))
		{
			$forum['deleteAnyPost'] = true;
		}

		if (!empty($modPerms['can_delete_content']) || !empty($modPerms['can_delete_post']))
		{
			$forum['hardDeleteAnyPost'] = true;
		}

		if (!empty($modPerms['can_hide_content']) || !empty($modPerms['can_hide_topic']))
		{
			$forum['deleteAnyThread'] = true;
		}

		if (!empty($modPerms['can_delete_content']) || !empty($modPerms['can_delete_topic']))
		{
			$forum['hardDeleteAnyThread'] = true;
		}

		if (!empty($modPerms['can_unhide_content'])
			|| !empty($modPerms['can_unhide_topic'])
			|| !empty($modPerms['can_unhide_post'])
		)
		{
			$forum['undelete'] = true;
		}

		if (!empty($modPerms['can_view_hidden_content'])
			|| !empty($moderator['can_view_hidden_topic'])
			|| !empty($modPerms['can_view_hidden_post'])
		)
		{
			$forum['viewDeleted'] = true;
		}

		return [
			'general' => $general,
			'forum' => $forum,
		];
	}

	// ########################### STEP: THREADS ###############################

	public function getStepEndThreads()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(tid)
			FROM forums_topics
		") ?: 0;
	}

	public function stepThreads(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$threads = $this->sourceDb->fetchAll("
				SELECT
					topics.*,
					IF (members.name IS NULL, topics.starter_name, members.name) AS starter_name,
					IF (lastposters.name IS NULL, topics.last_poster_name, lastposters.name) AS last_poster_name
				FROM forums_topics AS topics FORCE INDEX (PRIMARY)
				LEFT JOIN core_members AS members ON
					(topics.starter_id = members.member_id)
				LEFT JOIN core_members AS lastposters ON
					(topics.last_poster_id = lastposters.member_id)
				INNER JOIN forums_forums AS forums ON
					(topics.forum_id = forums.id AND forums.redirect_on = 0 AND forums.sub_can_post = 1)
				WHERE topics.tid > ? AND topics.tid <= ?
					AND topics.state <> 'link'
				ORDER BY topics.tid
				LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$threads)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		$mapNodeIds = [];

		foreach ($threads AS $thread)
		{
			$mapUserIds[] = $thread['starter_id'];
			$mapUserIds[] = $thread['last_poster_id'];
			$mapNodeIds[] = $thread['forum_id'];
		}

		$this->lookup('user', $mapUserIds);
		$this->lookup('node', $mapNodeIds);

		foreach ($threads AS $thread)
		{
			$oldId = $thread['tid'];
			$state->startAfter = $oldId;

			$nodeId = $this->lookupId('node', $thread['forum_id']);
			if (!$nodeId)
			{
				continue;
			}

			$userId = $this->lookupId('user', $thread['starter_id'], 0);
			$lastPostUserId = $this->lookupId('user', $thread['last_poster_id'], 0);

			/** @var Thread $import */
			$import = $this->newHandler(Thread::class);

			switch ($thread['approved'])
			{
				case 0:  $discussionState = 'moderated'; break;
				case -1: $discussionState = 'deleted';   break;
				case 2:  $discussionState = 'deleted';   break;
				default: $discussionState = 'visible';   break;
			}

			$import->bulkSet([
				'title' => $thread['title'],
				'reply_count' => $thread['posts'],
				'view_count' => $thread['views'],
				'post_date' => $thread['start_date'],
				'discussion_state' => $discussionState,
				'discussion_open' => ($thread['state'] == 'open'),
				'sticky' => boolval($thread['pinned']),
				'node_id' => $nodeId,
				'user_id' => $userId,
				'username' => $thread['starter_name'],
				'last_post_date' => $thread['last_post'],
				'last_post_user_id' => $lastPostUserId,
				'last_post_username' => $thread['last_poster_name'],
			]);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: POSTS ###############################

	public function getStepEndPosts()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(pid)
			FROM forums_posts
		") ?: 0;
	}

	public function stepPosts(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$posts = $this->sourceDb->fetchAll("
			SELECT posts.*,
				IF (members.name IS NULL, posts.author_name, members.name) AS author_name
			FROM forums_posts AS posts
			LEFT JOIN core_members AS members ON
				(posts.author_id = members.member_id)
			WHERE posts.pid > ? AND posts.pid <= ?
			ORDER BY posts.pid
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$posts)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		$mapThreadIs = [];

		foreach ($posts AS $post)
		{
			$mapUserIds[] = $post['author_id'];
			$mapThreadIs[] = $post['topic_id'];
		}

		$this->lookup('user', $mapUserIds);
		$this->lookup('thread', $mapThreadIs);

		foreach ($posts AS $post)
		{
			$oldId = $post['pid'];
			$state->startAfter = $oldId;

			$threadId = $this->lookupId('thread', $post['topic_id']);

			if (!$threadId)
			{
				continue;
			}

			$userId = $this->lookupId('user', $post['author_id'], 0);

			/** @var Post $import */
			$import = $this->newHandler(Post::class);

			switch ($post['queued'])
			{
				case 1:  $messageState = 'moderated'; break;
				case -1: $messageState = 'deleted';   break;
				default: $messageState = 'visible';   break;
			}

			$message = Ips::convertContentToBbCode($post['post'], 'post');

			$position = $this->db()->fetchOne("
				SELECT MAX(position)
				FROM xf_post
				WHERE thread_id = ?
			", $threadId) ?: 0;

			if ($messageState == 'visible')
			{
				$position++;
			}

			$import->bulkSet([
				'message_state' => $messageState,
				'post_date' => $post['post_date'],
				'thread_id' => $threadId,
				'user_id' => $userId,
				'username' => $post['author_name'],
				'last_edit_date' => $post['edit_time'],
				'position' => $position,
			]);

			$import->setLoggedIp($post['ip_address']);

			$helper = $this->getHelper();

			$quotedPostIds = $helper->getQuotedContentIds($message, 'post');
			$mentionedUserIds = $helper->getMentionedUserIds($message);

			$this->lookup('post', $quotedPostIds);
			$this->lookup('user', $mentionedUserIds);

			$message = $helper->rewriteQuotesInBbCode($message, 'post');
			$message = $helper->rewriteMentionsInBbCode($message);
			$import->message = $message;

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: ARCHIVE POSTS ###############################

	public function getStepEndArchivePosts()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(archive_id)
			FROM forums_archive_posts
		") ?: 0;
	}

	public function stepArchivePosts(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$posts = $this->sourceDb->fetchAll("
			SELECT posts.*,
				IF (members.name IS NULL, posts.archive_author_name, members.name) AS archive_author_name
			FROM forums_archive_posts AS posts
			LEFT JOIN core_members AS members ON
				(posts.archive_author_id = members.member_id)
			WHERE posts.archive_id > ? AND posts.archive_id <= ?
			ORDER BY posts.archive_id
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$posts)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		$mapThreadIs = [];

		foreach ($posts AS $post)
		{
			$mapUserIds[] = $post['archive_author_id'];
			$mapThreadIs[] = $post['archive_topic_id'];
		}

		$this->lookup('user', $mapUserIds);
		$this->lookup('thread', $mapThreadIs);

		foreach ($posts AS $post)
		{
			$oldId = $post['archive_id'];
			$state->startAfter = $oldId;

			$threadId = $this->lookupId('thread', $post['archive_topic_id']);

			if (!$threadId)
			{
				continue;
			}

			$userId = $this->lookupId('user', $post['archive_author_id'], 0);

			/** @var Post $import */
			$import = $this->newHandler(Post::class);

			switch ($post['archive_queued'])
			{
				case 1:  $messageState = 'moderated'; break;
				case -1: $messageState = 'deleted';   break;
				default: $messageState = 'visible';   break;
			}

			$message = Ips::convertContentToBbCode($post['archive_content'], 'post');

			$position = $this->db()->fetchOne("
				SELECT MAX(position)
				FROM xf_post
				WHERE thread_id = ?
			", $threadId) ?: 0;

			if ($messageState == 'visible')
			{
				$position++;
			}

			$import->bulkSet([
				'message_state' => $messageState,
				'post_date' => $post['archive_content_date'],
				'thread_id' => $threadId,
				'user_id' => $userId,
				'username' => $post['archive_author_name'],
				'last_edit_date' => $post['archive_edit_time'],
				'position' => $position,
			]);

			$import->setLoggedIp($post['archive_ip_address']);

			$helper = $this->getHelper();

			$quotedPostIds = $helper->getQuotedContentIds($message, 'post');
			$mentionedUserIds = $helper->getMentionedUserIds($message);

			$this->lookup('post', $quotedPostIds);
			$this->lookup('user', $mentionedUserIds);

			$message = $helper->rewriteQuotesInBbCode($message, 'post');
			$message = $helper->rewriteMentionsInBbCode($message);
			$import->message = $message;

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: TAGS ###############################

	public function getStepEndContentTags()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(tag_id)
			FROM core_tags
			WHERE tag_meta_app = 'forums'
			AND tag_meta_area = 'forums'
		") ?: 0;
	}

	public function stepContentTags(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$tags = $this->sourceDb->fetchAll("
			SELECT *
			FROM core_tags
			WHERE tag_id > ? AND tag_id <= ?
			AND tag_meta_app = 'forums' AND tag_meta_area = 'forums'
			ORDER BY tag_id
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$tags)
		{
			return $state->complete();
		}

		$mapUserIds = array_column($tags, 'tag_member_id');
		$mapThreadIds = array_column($tags, 'tag_meta_id');

		$this->lookup('user', $mapUserIds);
		$this->lookup('thread', $mapThreadIds);

		/** @var Tag $tagHelper */
		$tagHelper = $this->getDataHelper(Tag::class);

		foreach ($tags AS $tag)
		{
			$state->startAfter = $tag['tag_id'];

			if (!$newThreadId = $this->lookupId('thread', $tag['tag_meta_id']))
			{
				continue;
			}

			$thread = $this->em()->find(\XF\Entity\Thread::class, $newThreadId);
			if (!$thread)
			{
				continue;
			}

			$userId = $this->lookupId('user', $tag['tag_member_id'], 0);

			$tagText = trim($tag['tag_text']);

			$newId = $tagHelper->importTag($tagText, 'thread', $newThreadId, [
				'add_user_id' => $userId,
				'add_date' => $tag['tag_added'],
				'visible' => $thread->discussion_state == 'visible',
				'content_date' => $thread->post_date,
			]);

			if ($newId)
			{
				$state->imported++;
			}

			$this->em()->detachEntity($thread);

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: POLLS ###############################

	public function getStepEndPolls()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(pid)
			FROM core_polls
		") ?: 0;
	}

	public function stepPolls(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$polls = $this->sourceDb->fetchAllKeyed("
			SELECT poll.*, topic.tid
			FROM core_polls AS poll
			INNER JOIN forums_topics AS topic ON (topic.poll_state = poll.pid)
			WHERE poll.pid > ? AND poll.pid <= ?
			ORDER BY poll.pid
			LIMIT {$limit}
		", 'pid', [$state->startAfter, $state->end]);

		if (!$polls)
		{
			return $state->complete();
		}

		$pollsCompleted = [];

		$this->lookup('thread', array_column($polls, 'tid'));

		foreach ($polls AS $oldId => $poll)
		{
			$state->startAfter = $oldId;

			$newThreadId = $this->lookupId('thread', $poll['tid']);

			if (!$newThreadId)
			{
				continue;
			}

			if (array_key_exists($oldId, $pollsCompleted))
			{
				// poll id in the thread table isn't unique, so use this to avoid duplication
				continue;
			}

			$pollsCompleted[$oldId] = true;

			$questions = json_decode($poll['choices'], true);

			$question = $stepConfig['which_question'] == 'newest'
				? end($questions)
				: reset($questions);

			if ($poll['poll_close_date'] == -1)
			{
				$poll['poll_close_date'] = 0;
			}

			/** @var Poll $import */
			$import = $this->newHandler(Poll::class);

			if (!isset($question['question']))
			{
				continue;
			}

			$import->set('question', $question['question']);
			$import->bulkSet([
				'content_type' => 'thread',
				'content_id' => $newThreadId,
				'public_votes' => $poll['poll_view_voters'],
				'max_votes' => empty($question['multi']) ? 1 : 0,
				'close_date' => $poll['poll_close_date'],
			]);

			$importResponses = [];

			foreach ($question['choice'] AS $i => $responseText)
			{
				/** @var PollResponse $importResponse */
				$importResponse = $this->newHandler(PollResponse::class);
				$importResponse->preventRetainIds();
				$importResponse->response = $responseText;

				$importResponses[$i] = $importResponse;

				$import->addResponse($i, $importResponse);
			}

			$votes = $this->sourceDb->fetchAll("
				SELECT member_id, vote_date, member_choices
				FROM core_voters
				WHERE poll = ?
				ORDER BY vid
			", $oldId);

			$this->lookup('user', array_column($votes, 'member_id'));

			foreach ($votes AS $vote)
			{
				if (!$voteUserId = $this->lookupId('user', $vote['member_id']))
				{
					continue;
				}

				$allChoices = @json_decode($vote['member_choices'], true);

				if (!$allChoices)
				{
					continue;
				}

				$voteChoices = $stepConfig['which_question'] == 'newest'
					? end($allChoices)
					: reset($allChoices);

				if (!is_array($voteChoices))
				{
					$voteChoices = [$voteChoices];
				}

				foreach ($voteChoices AS $i)
				{
					if (!isset($importResponses[$i]))
					{
						continue;
					}

					$importResponses[$i]->addVote($voteUserId, $vote['vote_date']);
				}
			}

			if ($newId = $import->save($oldId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: ATTACHMENTS ###############################

	protected function getContentTypeMap()
	{
		return [
			'forums_Forums' => ['thread', 'post'],
			'core_Messaging' => ['conversation', 'conversation_message'],
		];
	}

	public function getStepEndAttachments()
	{
		$locationsQuoted = $this->sourceDb->quote(
			array_keys($this->getContentTypeMap())
		);

		return $this->sourceDb->fetchOne("
			SELECT MAX(attach.attach_id)
			FROM core_attachments AS attach
			INNER JOIN core_attachments_map AS map ON
				(attach.attach_id = map.attachment_id)
			WHERE map.location_key IN($locationsQuoted)
		") ?: 0;
	}

	public function stepAttachments(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$contentTypeMap = $this->getContentTypeMap();

		$locationsQuoted = $this->sourceDb->quote(
			array_keys($contentTypeMap)
		);

		$attachments = $this->sourceDb->fetchAll("
			SELECT
				attach.attach_id, attach.attach_date, attach.attach_hits,
				attach.attach_file,
			    attach.attach_location,
				attach.attach_member_id,
			    map.location_key, map.id1, map.id2
			FROM core_attachments AS attach
			INNER JOIN core_attachments_map AS map ON
				(attach.attach_id = map.attachment_id)
			WHERE attach.attach_id > ? AND attach.attach_id <= ?
				AND map.location_key IN($locationsQuoted)
			ORDER BY attach.attach_id
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$attachments)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		$mapContentIds = [];

		foreach ($attachments AS $attachment)
		{
			$mapUserIds[] = $attachment['attach_member_id'];

			[$parentType, $contentType] = $contentTypeMap[$attachment['location_key']];
			$mapContentIds[$parentType][] = $attachment['id1'];
			$mapContentIds[$contentType][] = $attachment['id2'];
		}

		$this->lookup('user', $mapUserIds);

		foreach ($mapContentIds AS $contentType => $contentIds)
		{
			$this->lookup($contentType, $contentIds);
		}

		foreach ($attachments AS $attachment)
		{
			$oldId = $attachment['attach_id'];
			$state->startAfter = $oldId;

			$userId = $this->lookupId('user', $attachment['attach_member_id'], 0);

			[$parentType, $contentType] = $contentTypeMap[$attachment['location_key']];

			$parentContentId = $this->lookupId($parentType, $attachment['id1']);
			$contentId = $this->lookupId($contentType, $attachment['id2']);

			if (!$parentContentId || !$contentId)
			{
				continue;
			}

			$sourceFile = sprintf(
				'%s/uploads/%s',
				$this->baseConfig['ips_path'],
				$attachment['attach_location']
			);

			if (!file_exists($sourceFile) || !is_readable($sourceFile))
			{
				continue;
			}

			/** @var Attachment $import */
			$import = $this->newHandler(Attachment::class);
			$import->bulkSet([
				'content_type' => $contentType,
				'content_id'   => $contentId,
				'attach_date'  => $attachment['attach_date'],
				'view_count'   => $attachment['attach_hits'],
				'unassociated' => false,
			]);
			$import->setDataExtra('upload_date', $attachment['attach_date']);
			$import->setDataUserId($userId);
			$import->setSourceFile($sourceFile, $attachment['attach_file']);
			$import->setContainerCallback([$this, 'rewriteEmbeddedAttachments']);

			if ($newId = $import->save($oldId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: REACTIONS ###############################

	public function stepReactions(StepState $state)
	{
		$mappableTypes = [
			'reactions/react_like.png' => 1,
			'reactions/react_haha.png' => 3,
			'reactions/react_sad.png' => 5,
		];

		$reactions = $this->sourceDb->fetchAll("
			SELECT reaction.*, lang_title.word_custom AS title
			FROM core_reactions AS reaction
			LEFT JOIN core_sys_lang_words AS lang_title ON
				(lang_title.lang_id = 1 AND lang_title.word_key = CONCAT('reaction_title_', reaction.reaction_id))
			ORDER BY reaction.reaction_id
		");

		foreach ($reactions AS $reaction)
		{
			$oldId = $reaction['reaction_id'];

			if (isset($mappableTypes[$reaction['reaction_icon']]))
			{
				// same or similar to our default reactions so do not import and map to ours
				$this->logHandler(Reaction::class, $oldId, $mappableTypes[$reaction['reaction_icon']]);
			}
			else
			{
				/** @var Reaction $import */
				$import = $this->newHandler(Reaction::class);
				$import->preventRetainIds(); // necessary due to default reactions

				$import->bulkSet($this->mapXfKeys($reaction, [
					'reaction_score' => 'reaction_value',
					'display_order' => 'reaction_position',
					'active' => 'reaction_enabled',
				]));
				$import->setTitle($reaction['title']);

				$sourceFile = sprintf(
					'%s/uploads/%s',
					$this->baseConfig['ips_path'],
					$reaction['reaction_icon']
				);

				if (file_exists($sourceFile) && is_readable($sourceFile))
				{
					$import->setSourceImagePath($sourceFile, $reaction['reaction_icon']);
				}

				$newId = $import->save($oldId);
				if ($newId)
				{
					$state->imported++;
				}
			}
		}

		return $state->complete();
	}

	// ########################### STEP: REACTION CONTENT ###############################

	public function getStepEndReactionContent()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(id)
			FROM core_reputation_index
			WHERE app = 'forums' AND type = 'pid' AND reaction > 0
		");
	}

	public function stepReactionContent(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$reactionRep = $this->sourceDb->fetchAll("
			SELECT rep.*, post.pid
			FROM core_reputation_index AS rep
			INNER JOIN forums_posts AS post ON
				(post.pid = rep.type_id)
			WHERE rep.id > ? AND rep.id <= ?
			AND rep.app = 'forums' AND rep.type = 'pid' AND rep.reaction > 0
			ORDER BY rep.id
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$reactionRep)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		$mapPostIds = [];
		$mapReactionIds = [];

		foreach ($reactionRep AS $reaction)
		{
			$mapUserIds[] = $reaction['member_id'];
			$mapUserIds[] = $reaction['member_received'];
			$mapPostIds[] = $reaction['pid'];
			$mapReactionIds[] = $reaction['reaction'];
		}

		$this->lookup('user', $mapUserIds);
		$this->lookup('post', $mapPostIds);
		$this->lookup('reaction', $mapReactionIds);

		foreach ($reactionRep AS $reaction)
		{
			$oldId = $reaction['id'];
			$state->startAfter = $oldId;

			$reactionUserId = $this->lookupId('user', $reaction['member_id']);
			if (!$reactionUserId)
			{
				continue;
			}

			$postId = $this->lookupId('post', $reaction['pid']);
			if (!$postId)
			{
				continue;
			}

			$reactionId = $this->lookupId('reaction', $reaction['reaction']);
			if (!$reactionId)
			{
				continue;
			}

			$exists = $this->em()->findOne(\XF\Entity\ReactionContent::class, [
				'content_type' => 'post',
				'content_id' => $postId,
				'reaction_user_id' => $reactionUserId,
			]);
			if ($exists)
			{
				// some evidence that records exist where content has been repped by
				// the same user multiple times so skip these if imported already
				continue;
			}

			/** @var ReactionContent $import */
			$import = $this->newHandler(ReactionContent::class);

			$import->reaction_date = $reaction['rep_date'];
			$import->setReactionId($reactionId);
			$import->content_type = 'post';
			$import->content_id = $postId;
			$import->reaction_user_id = $reactionUserId;
			$import->content_user_id = $this->lookupId('user', $reaction['member_received'], 0);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function decodeValue($value, $type)
	{
		switch ($type)
		{
			case 'list-comma-ips':
				return preg_split('/,/', $value, -1, PREG_SPLIT_NO_EMPTY);

			default:
				return parent::decodeValue($value, $type);
		}
	}
}
