<?php

namespace XFI\Import\Importer;

use XF\Db\Mysqli\Adapter;
use XF\Entity\Attachment;
use XF\Entity\User;
use XF\Import\Data\Category;
use XF\Import\Data\ConversationMaster;
use XF\Import\Data\ConversationMessage;
use XF\Import\Data\EntityEmulator;
use XF\Import\Data\Forum;
use XF\Import\Data\LinkForum;
use XF\Import\Data\Node;
use XF\Import\Data\Poll;
use XF\Import\Data\PollResponse;
use XF\Import\Data\Post;
use XF\Import\Data\Thread;
use XF\Import\Data\UserField;
use XF\Import\Data\UserGroup;
use XF\Import\DataHelper\Moderator;
use XF\Import\Importer\AbstractForumImporter;
use XF\Import\Importer\StepPostsTrait;
use XF\Import\StepState;
use XF\Mvc\Entity\Entity;
use XF\Timer;
use XF\Util\File;

use function array_key_exists, boolval, count, intval, is_array;

class PhpBb extends AbstractForumImporter
{
	use StepPostsTrait;

	/**
	 * @var Adapter
	 */
	protected $sourceDb;

	public static function getListInfo()
	{
		return [
			'target' => 'XenForo',
			'source' => 'phpBB 3.2, 3.3',
		];
	}

	protected function getBaseConfigDefault()
	{
		return [
			'db' => [
				'host' => '',
				'username' => '',
				'password' => '',
				'dbname' => '',
				'port' => 3306,
				'tablePrefix'   => '',
			],
			'avatar_path' => '',
			'attach_path'   => [],
			'lang_id' => 1,
		];
	}

	public function renderBaseConfigOptions(array $vars)
	{
		$vars['db'] = [
			'host' => $this->app->config['db']['host'],
			'port' => $this->app->config['db']['port'],
			'username' => $this->app->config['db']['username'],
		];
		return $this->app->templater()->renderTemplate('admin:xfi_import_config_phpbb', $vars);
	}

	public function validateBaseConfig(array &$baseConfig, array &$errors)
	{
		$baseConfig['db']['tablePrefix'] = preg_replace('/[^a-z0-9_]/i', '', $baseConfig['db']['tablePrefix']);

		$fullConfig = array_replace_recursive($this->getBaseConfigDefault(), $baseConfig);
		$missingFields = false;

		if ($fullConfig['db']['host'])
		{
			$validDbConnection = false;

			try
			{
				$sourceDb = new Adapter($fullConfig['db'], true);
				$sourceDb->getConnection();
				$validDbConnection = true;
			}
			catch (\XF\Db\Exception $e)
			{
				$errors[] = \XF::phrase('source_database_connection_details_not_correct_x', ['message' => $e->getMessage()]);
			}

			if ($validDbConnection)
			{
				try
				{
					$sourceDb->fetchOne("
						SELECT user_id
						FROM users
						LIMIT 1
					");

					$language = $sourceDb->fetchOne("
						SELECT config_value
						FROM config
						WHERE config_name = 'default_lang'
					");

					$languageId = $sourceDb->fetchOne("
						SELECT lang_id
						FROM lang
						WHERE lang_iso = ?
					", $language);

					if ($languageId)
					{
						$baseConfig['lang_id'] = $languageId;
					}
				}
				catch (\XF\Db\Exception $e)
				{
					if ($fullConfig['db']['dbname'] === '')
					{
						$errors[] = \XF::phrase('please_enter_database_name');
					}
					else
					{
						$errors[] = \XF::phrase('table_prefix_or_database_name_is_not_correct');
					}
				}
			}
			else
			{
				$missingFields = true;
			}
		}

		if ($fullConfig['avatar_path'])
		{
			$path = rtrim($fullConfig['avatar_path'], '/\\ ');

			if (!file_exists($path) || !is_dir($path))
			{
				$errors[] = \XF::phrase('directory_x_not_found_is_not_readable', ['dir' => $path]);
			}

			$baseConfig['avatar_path'] = $path;
		}
		else
		{
			$missingFields = true;
		}

		if ($fullConfig['attach_path'])
		{
			$path = rtrim($fullConfig['attach_path'], '/\\ ');

			if (!file_exists($path) || !is_dir($path))
			{
				$errors[] = \XF::phrase('directory_x_not_found_is_not_readable', ['dir' => $path]);
			}

			$baseConfig['attach_path'] = $path;
		}
		else
		{
			$missingFields = true;
		}

		if ($missingFields)
		{
			$errors[] = \XF::phrase('please_complete_required_fields');
		}

		return $errors ? false : true;
	}

	protected function getStepConfigDefault()
	{
		return [
			'users' => [
				'merge_email' => true,
				'merge_name'  => false,
			],
		];
	}

	public function renderStepConfigOptions(array $vars)
	{
		$vars['stepConfig'] = $this->getStepConfigDefault();
		return $this->app->templater()->renderTemplate('admin:xfi_import_step_config_phpbb', $vars);
	}

	public function validateStepConfig(array $steps, array &$stepConfig, array &$errors)
	{
		return true;
	}

	public function getSteps()
	{
		return [
			'userGroups' => [
				'title' => \XF::phrase('user_groups'),
			],
			'userFields' => [
				'title' => \XF::phrase('custom_user_fields'),
			],
			'users' => [
				'title' => \XF::phrase('users'),
				'depends' => ['userGroups', 'userFields'],
			],
			'privateMessages' => [
				'title' => \XF::phrase('xfi_import_private_messages'),
				'depends' => ['users'],
			],
			'forums' => [
				'title' => \XF::phrase('forums'),
				'depends' => ['userGroups'],
			],
			'moderators' => [
				'title' => \XF::phrase('moderators'),
				'depends' => ['forums', 'users'],
			],
			'threads' => [
				'title' => \XF::phrase('threads'),
				'depends' => ['forums', 'users'],
				'force' => ['posts'],
			],
			'posts' => [
				'title' => \XF::phrase('posts'),
				'depends' => ['threads'],
			],
			'polls' => [
				'title' => \XF::phrase('thread_polls'),
				'depends' => ['threads'],
			],
			'attachments' => [
				'title' => \XF::phrase('attachments'),
				'depends' => ['threads'],
			],
		];
	}

	protected function doInitializeSource()
	{
		$dbConfig = $this->baseConfig['db'];
		$this->sourceDb = new Adapter($dbConfig, true);
	}

	// ############################## STEP: USER GROUPS #########################

	public function stepUserGroups(StepState $state, array $stepConfig)
	{
		$groups = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM groups
			ORDER BY group_id
		", 'group_id');

		foreach ($groups AS $oldId => $group)
		{
			$permissions = $this->calculateGroupPerms($oldId);
			$titlePriority = 5;

			$groupMap = [
				1 => User::GROUP_GUEST,
				6 => User::GROUP_GUEST,

				2 => User::GROUP_REG,
				3 => User::GROUP_REG,

				4 => User::GROUP_MOD,

				5 => User::GROUP_ADMIN,
			];

			if (array_key_exists($oldId, $groupMap))
			{
				// don't import the group, just map it to one of our defaults
				$this->logHandler(UserGroup::class, $oldId, $groupMap[$oldId]);
			}
			else
			{
				if ($oldId == 4) // super mods
				{
					$titlePriority = 910;
				}

				/** @var UserGroup $import */
				$import = $this->newHandler(UserGroup::class);
				$import->title = $group['group_name'];
				$import->display_style_priority = $titlePriority;
				$import->setPermissions($permissions);
				$import->save($oldId);
			}

			$state->imported++;
		}

		return $state->complete();
	}

	protected function calculateGroupPerms($groupId)
	{
		$oldPerms = $this->getGroupPermissions($groupId, 0);
		$perms = [];

		// no equivalents
		$perms['general']['view'] = 'allow';
		$perms['general']['viewNode'] = 'allow';
		$perms['general']['submitWithoutApproval'] = 'allow';
		$perms['general']['report'] = 'allow';
		$perms['forum']['viewOthers'] = 'allow';
		$perms['forum']['viewContent'] = 'allow';
		$perms['forum']['react'] = 'allow';
		$perms['forum']['votePoll'] = 'allow';
		$perms['conversation']['receive'] = 'allow';

		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'editSignature', 'u_sig');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'bypassFloodCheck', 'u_ignoreflood');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'bypassUserPrivacy', 'u_viewonline');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'viewProfile', 'u_viewprofile');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'viewMemberList', 'u_viewprofile');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'search', 'u_search');
		$this->convertOldPermissionValue($perms, $oldPerms, 'forum', 'uploadAttachment', 'u_attach');
		$this->convertOldPermissionValue($perms, $oldPerms, 'forum', 'viewAttachment', 'u_download');
		$this->convertOldPermissionValue($perms, $oldPerms, 'avatar', 'allowed', 'u_chgavatar');
		$this->convertOldPermissionValue($perms, $oldPerms, 'conversation', 'start', 'u_sendpm');
		$this->convertOldPermissionValue($perms, $oldPerms, 'conversation', 'editOwnMessage', 'u_pm_edit');

		if ($this->hasOldPermission($oldPerms, 'u_pm_edit'))
		{
			$perms['conversation']['editOwnMessageTimeLimit'] = 5;
		}
		if ($this->hasOldPermission($oldPerms, 'u_viewprofile'))
		{
			$perms['profilePost']['view'] = 'allow';
			$perms['profilePost']['react'] = 'allow';
			$perms['profilePost']['manageOwn'] = 'allow';
			$perms['profilePost']['deleteOwn'] = 'allow';
			$perms['profilePost']['post'] = 'allow';
			$perms['profilePost']['comment'] = 'allow';
			$perms['profilePost']['editOwn'] = 'allow';
		}

		return $perms;
	}

	protected function getGroupPermissions($ids, $forumId = 0)
	{
		$perms = [];

		if (!is_array($ids))
		{
			$ids = [$ids];
		}

		if (!$ids)
		{
			return [];
		}

		$groupPermsResult = $this->sourceDb->query('
			SELECT o.auth_option, g.auth_setting
			FROM acl_groups AS g
			INNER JOIN acl_options AS o ON (g.auth_option_id = o.auth_option_id)
			WHERE g.group_id IN (' . $this->sourceDb->quote($ids) . ')
				AND g.auth_option_id > 0
				AND g.forum_id = ' . $this->sourceDb->quote($forumId) . '
		');
		while ($perm = $groupPermsResult->fetch())
		{
			$this->mergePermissions($perms, $perm);
		}

		$rolePermsResult = $this->sourceDb->query('
			SELECT o.auth_option, r.auth_setting
			FROM acl_groups AS g
			INNER JOIN acl_roles_data AS r ON (g.auth_role_id = r.role_id)
			INNER JOIN acl_options AS o ON (r.auth_option_id = o.auth_option_id)
			WHERE g.group_id IN (' . $this->sourceDb->quote($ids) . ')
				AND g.auth_role_id > 0
				AND g.forum_id = ' . $this->sourceDb->quote($forumId) . '
		');
		while ($perm = $rolePermsResult->fetch())
		{
			$this->mergePermissions($perms, $perm);
		}

		return $perms;
	}

	protected function getUserPermissions($id, $forumId = 0)
	{
		$perms = [];

		$userPermsResult = $this->sourceDb->query('
			SELECT o.auth_option, u.auth_setting
			FROM acl_users AS u
			INNER JOIN acl_options AS o ON (u.auth_option_id = o.auth_option_id)
			WHERE u.user_id = ' . $this->sourceDb->quote($id) . '
				AND u.auth_option_id > 0
				AND u.forum_id = ' . $this->sourceDb->quote($forumId) . '
		');
		while ($perm = $userPermsResult->fetch())
		{
			$this->mergePermissions($perms, $perm);
		}

		$rolePermsResult = $this->sourceDb->query('
			SELECT o.auth_option, r.auth_setting
			FROM acl_users AS u
			INNER JOIN acl_roles_data AS r ON (u.auth_role_id = r.role_id)
			INNER JOIN acl_options AS o ON (r.auth_option_id = o.auth_option_id)
			WHERE u.user_id = ' . $this->sourceDb->quote($id) . '
				AND u.auth_role_id > 0
				AND u.forum_id = ' . $this->sourceDb->quote($forumId) . '
		');
		while ($perm = $rolePermsResult->fetch())
		{
			$this->mergePermissions($perms, $perm);
		}

		$groups = $this->sourceDb->fetchAllColumn(
			'
			SELECT group_id
			FROM user_group
			WHERE user_id = ' . $this->sourceDb->quote($id)
		);
		$groupPerms = $this->getGroupPermissions($groups, $forumId);
		foreach ($groupPerms AS $permId => $permSetting)
		{
			$perm = ['auth_option' => $permId, 'auth_setting' => $permSetting];
			$this->mergePermissions($perms, $perm);
		}

		return $perms;
	}

	protected function mergePermissions(array &$perms, array $perm)
	{
		if (!isset($perms[$perm['auth_option']]))
		{
			$perms[$perm['auth_option']] = intval($perm['auth_setting']);
		}
		else if ($perms[$perm['auth_option']] == 0 || $perm['auth_option'] == 0)
		{
			$perms[$perm['auth_option']] = 0;
		}
		else if ($perm['auth_option'] == 1)
		{
			$perms[$perm['auth_option']] = 1;
		}
		else
		{
			$perms[$perm['auth_option']] = -1;
		}
	}

	protected function hasOldPermission(array $oldPerms, $oldId)
	{
		return isset($oldPerms[$oldId]) && $oldPerms[$oldId] == 1;
	}

	protected function convertOldPermissionValue(array &$outputPerms, array $oldPerms, $newGroup, $newId, $oldId, $allow = 'allow')
	{
		if (!isset($oldPerms[$oldId]) || $oldPerms[$oldId] == -1)
		{
			return;
		}

		if ($oldPerms[$oldId] == 0)
		{
			$outputPerms[$newGroup][$newId] = 'deny';
		}
		else if ($oldPerms[$oldId] == 1)
		{
			$outputPerms[$newGroup][$newId] = $allow;
		}
	}

	// ############################## STEP: USER FIELDS #########################

	public function stepUserFields(StepState $state, array $stepConfig)
	{
		$fields = $this->sourceDb->fetchAllKeyed("
			SELECT pf.*, pl.*
			FROM profile_fields AS pf
			INNER JOIN profile_lang AS pl ON
				(pf.field_id = pl.field_id AND pl.lang_id = ?)
			ORDER BY pf.field_id
		", 'field_id', [$this->baseConfig['lang_id']]);

		$optionsGrouped = [];
		$options = $this->sourceDb->query("
			SELECT *
			FROM profile_fields_lang
			WHERE lang_id = ?
		", [$this->baseConfig['lang_id']]);

		while ($option = $options->fetch())
		{
			$optionsGrouped[$option['field_id']][$option['option_id']] = $option['lang_value'];
		}

		$choiceLookUps = [];
		$existingFields = $this->db()->fetchPairs("SELECT field_id, field_id FROM xf_user_field");

		foreach ($fields AS $oldId => $field)
		{
			$fieldId = $this->convertToUniqueId($field['field_name'], $existingFields, 25);

			$data = [
				'field_id' => $fieldId,
				'display_order' => $field['field_order'],
				'max_length' => $field['field_maxlen'],
				'viewable_profile' => !$field['field_no_view'],
				'user_editable' => $field['field_show_profile'] ? 'yes' : 'never',
				'viewable_message' => (isset($field['field_show_on_vt']) ? boolval($field['field_show_on_vt']) : false),
				'show_registration' => $field['field_show_on_reg'],
				'required' => boolval($field['field_required']),
			];

			if ($field['field_validation'] && $field['field_validation'] != '.*')
			{
				$data['match_type'] = 'regex';
				$data['match_params'] = [
					'regex' => '^' . $field['field_validation'] . '$',
				];
			}

			switch ($field['field_type'])
			{
				case 'profilefields.type.int': // numbers
					$data['field_type'] = 'textbox';
					$data['match_type'] = 'number';
					if ($field['field_minlen'] || $field['field_maxlen'])
					{
						$data['match_params'] = [
							'number_min' => $field['field_minlen'],
							'number_max' => $field['field_maxlen'],
						];
					}
					break;

				case 'profilefields.type.string': // text box
					$data['field_type'] = 'textbox';
					break;

				case 'profilefields.type.text': // text area
					$data['field_type'] = 'textarea';
					break;

				case 'profilefields.type.bool': // boolean
				case 'profilefields.type.dropdown': // drop down
					$data['field_type'] = ($field['field_type'] == 'profilefields.type.bool' ? 'radio' : 'select');
					if (empty($optionsGrouped[$field['field_id']]))
					{
						break;
					}
					$data['field_choices'] = $optionsGrouped[$field['field_id']];
					$choiceLookUps[$field['field_name']] = $optionsGrouped[$field['field_id']];
					break;

				case 'profilefields.type.date': // date
					$data['field_type'] = 'textbox';
					$data['match_type'] = 'date';
					break;

				case 'profilefields.type.url': // url
					$data['field_type'] = 'textbox';
					$data['match_type'] = 'url';
					break;

				default:
					$data['field_type'] = 'textbox';
					break;
			}

			$title = $this->convertToUtf8($field['lang_name'], null, true);
			$description = $this->convertToUtf8($field['lang_explain'], null, true);

			/** @var UserField $import */
			$import = $this->newHandler(UserField::class);
			$import->setTitle($title, $description);
			$import->bulkSet($data);

			$import->save($oldId);

			$state->imported++;
		}

		$this->session->extra['profileFieldChoices'] = $choiceLookUps;

		return $state->complete();
	}

	// ############################## STEP: USERS #############################

	public function getStepEndUsers()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(user_id) FROM users") ?: 0;
	}

	public function stepUsers(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$fields = $this->sourceDb->fetchPairs("
			SELECT pf.field_name, pf.field_name
			FROM profile_fields AS pf
		");

		$websiteColSql = isset($fields['phpbb_website']) ? 'pfd.pf_phpbb_website' : "''";
		$interestsColSql = isset($fields['phpbb_interests']) ? 'pfd.pf_phpbb_interests' : "''";
		$locationColSql = isset($fields['phpbb_location']) ? 'pfd.pf_phpbb_location' : "''";

		$users = $this->sourceDb->fetchAllKeyed("
			SELECT users.*, pfd.*, ban.*, users.user_id, 1 AS user_dst,
				{$websiteColSql} AS user_website,
				{$interestsColSql} AS user_interests,
				{$locationColSql} AS user_from
			FROM users AS users
			LEFT JOIN profile_fields_data AS pfd ON
				(pfd.user_id = users.user_id)
			LEFT JOIN banlist AS ban ON
				(ban.ban_userid = users.user_id AND (ban.ban_end = 0 OR ban.ban_end > ?))
			WHERE users.user_id > ? AND users.user_id <= ?
				AND users.user_type <> 2
			ORDER BY users.user_id
			LIMIT {$limit}
		", 'user_id', [time(), $state->startAfter, $state->end]);

		if (!$users)
		{
			return $state->complete();
		}

		$this->typeMap('user_group');

		foreach ($users AS $oldId => $user)
		{
			$state->startAfter = $oldId;

			$import = $this->setupImportUser($user, $state, $stepConfig);
			if ($this->importUser($oldId, $import, $stepConfig))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected function setupImportUser(array $user, StepState $state, array $stepConfig)
	{
		/** @var \XF\Import\Data\User $import */
		$import = $this->newHandler(\XF\Import\Data\User::class);

		$userData = $this->mapXfKeys($user, [
			'username',
			'email' => 'user_email',
			'last_activity' => 'user_lastvisit',
			'register_date' => 'user_regdate',
			'message_count' => 'user_posts',
			'visible' => 'user_allow_viewonline',
		]);

		$import->bulkSetDirect('user', $userData);

		if (!isset($this->session->extra['userActivationSetting']))
		{
			$this->session->extra['userActivationSetting'] = $this->sourceDb->fetchOne("
				SELECT config_value
				FROM config
				WHERE config_name = 'require_activation'
			");
		}
		if (!isset($this->session->extra['userAvatarSalt']))
		{
			$this->session->extra['userAvatarSalt'] = $this->sourceDb->fetchOne("
				SELECT config_value
				FROM config
				WHERE config_name = 'avatar_salt'
			");
		}

		if ($user['group_id'] == 3) // coppa
		{
			$import->user_state = 'moderated';
		}
		else if ($user['user_type'] == 1)
		{
			switch ($user['user_inactive_reason'])
			{
				case 1:
					// INACTIVE_REGISTER - Newly registered account
					$import->user_state = ($this->session->extra['userActivationSetting'] == 2 ? 'moderated' : 'email_confirm');
					break;

				case 2:
					// INACTIVE_PROFILE - Profile details changed
					$import->user_state = 'email_confirm_edit';
					break;

				case 3:
					// INACTIVE_MANUAL - Account deactivated by administrator
					$import->user_state = 'disabled';
					break;

				case 4:
					// INACTIVE_REMIND - Forced user account reactivation
					$import->user_state = 'moderated';
					break;

				default:
					// account not active but reason unknown we'll treat as disabled
					$import->user_state = 'disabled';
					break;
			}
		}
		else
		{
			$import->user_state = 'valid';
		}

		// user groups
		$groups = $this->sourceDb->fetchAllColumn("
			SELECT group_id
			FROM user_group
			WHERE user_id = ?
				AND group_id <> ?
				AND user_pending = 0
		", [$user['user_id'], $user['group_id']]);

		$import->user_group_id = $this->lookupId('user_group', $user['group_id'], User::GROUP_REG);
		if ($groups)
		{
			$import->secondary_group_ids = $this->lookup('user_group', $groups);
		}

		$import->is_admin = ($user['user_type'] == 3 ? 1 : 0);
		if ($import->is_admin)
		{
			// no real concept of admin permissions so import as super admins
			$adminData = [
				'is_super_admin' => true,
				'permission_cache' => [],
			];

			$import->setAdmin($adminData);
		}

		$import->is_banned = ($user['ban_id'] ? 1 : 0);

		if ($import->is_banned)
		{
			$import->setBan([
				'ban_user_id' => 0,
				'ban_date' => $user['ban_start'],
				'end_date' => $user['ban_end'],
				'user_reason' => $user['ban_give_reason'],
			]);
		}

		$useDst = (
			$user['user_dst'] ?? ($user['use_dst'] ?? 1)
		);
		try
		{
			$time = new \DateTime('now', new \DateTimeZone($user['user_timezone']));
			$offset = $time->getTimezone()->getOffset($time) / 60 / 60;
			$import->timezone = $this->getTimezoneFromOffset($offset, $useDst);
		}
		catch (\Exception $e)
		{
			$import->timezone = $this->getTimezoneFromOffset($user['user_timezone'], $useDst);
		}

		$import->setPasswordData('XF:PhpBb3', [
			'hash' => $user['user_password'],
		]);

		$import->setRegistrationIp($user['user_ip']);

		$import->website = $user['user_website'];
		$import->signature = $this->convertContentToBbCode($user['user_sig']);

		if ($user['user_birthday'])
		{
			$parts = explode('-', $user['user_birthday']);
			if (count($parts) == 3)
			{
				$import->dob_day = trim($parts[0]);
				$import->dob_month = trim($parts[1]);
				$import->dob_year = trim($parts[2]);
			}
		}

		switch ($user['user_avatar_type'])
		{
			case 'avatar.driver.upload':
				$avatarSalt = $this->session->extra['userAvatarSalt'];
				$userId = $user['user_id'];
				$extension = File::getFileExtension($user['user_avatar']);

				$originalPath = $this->baseConfig['avatar_path'] . "/{$avatarSalt}_{$userId}.{$extension}";
				if (!file_exists($originalPath))
				{
					break;
				}

				$import->setAvatarPath($originalPath);
				break;

			case 'avatar.driver.gravatar':
				$import->gravatar = $user['user_avatar'];
				break;

			case 'avatar.driver.remote':
				$avatarPath = $this->getImageFromUrl($user['user_avatar']);
				if ($avatarPath)
				{
					$import->setAvatarPath($avatarPath);
				}
				break;
		}

		$import->about = $user['user_interests'];
		$import->location = $user['user_from'];

		$choiceLookUps = $this->session->extra['profileFieldChoices'];
		$fieldValues = [];

		foreach ($this->typeMap('user_field') AS $fieldId => $newFieldId)
		{
			$fieldValue = '';

			if (isset($user["pf_$fieldId"]) && $user["pf_$fieldId"] !== '')
			{
				if (isset($choiceLookUps[$fieldId]))
				{
					$fieldInfo = $choiceLookUps[$fieldId];
					$fieldChoiceId = max(0, $user["pf_$fieldId"] - 1); // option ids are 0 keyed, values are 1 keyed

					if (isset($fieldInfo[$fieldChoiceId]))
					{
						$fieldValue = $fieldInfo[$fieldChoiceId];
					}
				}
				else
				{
					// set the field value directly
					$fieldValue = $user["pf_$fieldId"];
				}
			}

			$fieldValues[$newFieldId] = $fieldValue;
		}

		$import->setCustomFields($fieldValues);

		$optionData = $this->mapXfKeys($user, [
			'receive_admin_email' => 'user_allow_massemail',
			'email_on_conversation' => 'user_notify_pm',
			'push_on_conversation' => 'user_notify_pm',
		]);
		$optionData['content_show_signature'] = ((intval($user['user_options']) & 1 << 3) ? 1 : 0);

		$import->bulkSetDirect('option', $optionData);

		$import->allow_send_personal_conversation = ($user['user_allow_pm'] ? 'members' : 'none');

		return $import;
	}

	// ############################## STEP: PRIVATE MESSAGES #############################

	public function getStepEndPrivateMessages()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(msg_id) FROM privmsgs") ?: 0;
	}

	public function stepPrivateMessages(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$pms = $this->sourceDb->fetchAllKeyed("
			SELECT pms.*, users.username
			FROM privmsgs AS pms
			LEFT JOIN users AS users ON (pms.author_id = users.user_id)
			WHERE pms.msg_id > ? AND pms.msg_id <= ?
			ORDER BY pms.msg_id
			LIMIT {$limit}
		", 'msg_id', [$state->startAfter, $state->end]);

		if (!$pms)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		foreach ($pms AS $pm)
		{
			$mapUserIds[] = $pm['author_id'];
			$recipientUserIds = $this->getRecipientUserIds($pm);
			foreach ($recipientUserIds AS $recipient)
			{
				$mapUserIds[] = $recipient;
			}
		}

		$this->lookup('user', $mapUserIds);

		foreach ($pms AS $oldId => $pm)
		{
			$state->startAfter = $oldId;

			$newFromUserId = $this->lookupId('user', $pm['author_id']);
			if (!$newFromUserId)
			{
				continue;
			}

			$recipientUserIds = $this->getRecipientUserIds($pm);
			if (!$recipientUserIds)
			{
				continue;
			}

			$recipients = $this->sourceDb->fetchPairs("
				SELECT user_id, username
				FROM users
				WHERE user_id IN(" . $this->sourceDb->quote($recipientUserIds) . ")
			");
			if (!$recipients)
			{
				continue;
			}

			$users = [
				$pm['author_id'] => $pm['username'],
			] + $recipients;

			$unreadState = $this->sourceDb->fetchPairs("
				SELECT user_id, MIN(IF(folder_id < 0, 0, pm_unread))
				FROM privmsgs_to
				WHERE msg_id = ?
					AND pm_deleted = 0
				GROUP BY user_id
			", $oldId);

			/** @var ConversationMaster $import */
			$import = $this->newHandler(ConversationMaster::class);

			foreach ($users AS $userId => $username)
			{
				$newUserId = $this->lookupId('user', $userId);
				if (!$newUserId)
				{
					continue;
				}

				if (isset($unreadState[$userId]))
				{
					$lastReadDate = ($unreadState[$userId] ? 0 : $pm['message_time']);
					$deleted = false;
				}
				else
				{
					$lastReadDate = $pm['message_time'];
					$deleted = true;
				}

				$import->addRecipient($newUserId, $deleted ? 'deleted' : 'active', [
					'last_read_date' => $lastReadDate,
				]);
			}

			$fromUserName = $users[$pm['author_id']];

			$import->title = $pm['message_subject'];
			$import->user_id = $newFromUserId;
			$import->username = $fromUserName;
			$import->start_date = $pm['message_time'];

			/** @var ConversationMessage $messageImport */
			$messageImport = $this->newHandler(ConversationMessage::class);

			$messageImport->message_date = $pm['message_time'];
			$messageImport->user_id = $newFromUserId;
			$messageImport->username = $fromUserName;
			$messageImport->message = $this->convertContentToBbCode($pm['message_text']);

			$import->addMessage($oldId, $messageImport);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected function getRecipientUserIds(array $pm)
	{
		$recipientUserIds = [];

		if (preg_match_all('#u_(\d+)#', $pm['to_address'], $matches))
		{
			$recipientUserIds = $matches[1];
		}

		return $recipientUserIds;
	}

	// ############################## STEP: FORUMS #############################

	public function stepForums(StepState $state, array $stepConfig)
	{
		$forums = $this->sourceDb->fetchAll('
			SELECT *,
				forum_posts_approved AS forum_posts,
				forum_topics_approved AS forum_topics
			FROM forums
		');
		if (!$forums)
		{
			return $state->complete();
		}

		$nodeTree = [];
		foreach ($forums AS $forum)
		{
			$nodeTree[$forum['parent_id']][$forum['forum_id']] = $forum;
		}

		$state->imported = $this->importNodeTree($forums, $nodeTree);

		return $state->complete();
	}

	protected function importNodeTree(array $nodes, array $tree, $oldParentId = 0)
	{
		if (!isset($tree[$oldParentId]))
		{
			return 0;
		}

		$total = 0;

		foreach ($tree[$oldParentId] AS $node)
		{
			$oldNodeId = $node['forum_id'];

			/** @var Node $importNode */
			$importNode = $this->newHandler(Node::class);
			$importNode->bulkSet($this->mapXfKeys($node, [
				'display_order' => 'left_id',
			]));
			$importNode->title = $this->convertToUtf8($node['forum_name'], null, true);
			$importNode->description = $this->convertContentToBbCode($node['forum_desc'], true);
			$importNode->parent_node_id = $this->lookupId('node', $node['parent_id'], 0);

			if ($node['forum_type'] == 2)
			{
				$nodeTypeId = 'LinkForum';

				/** @var LinkForum $importType */
				$importType = $this->newHandler(LinkForum::class);
				$importType->link_url = $node['forum_link'];
			}
			else if ($node['forum_type'] == 1)
			{
				$nodeTypeId = 'Forum';

				/** @var Forum $importType */
				$importType = $this->newHandler(Forum::class);
				$importType->bulkSet($this->mapXfKeys($node, [
					'discussion_count' => 'forum_topics',
					'message_count' => 'forum_posts',
					'last_post_date' => 'forum_last_post_time',
					'last_post_username' => 'forum_last_poster_name',
					'last_thread_title' => 'forum_last_post_subject',
				]));
			}
			else
			{
				$nodeTypeId = 'Category';

				/** @var Category $importType */
				$importType = $this->newHandler(Category::class);
			}

			$importNode->setType($nodeTypeId, $importType);

			$newNodeId = $importNode->save($oldNodeId);
			if ($newNodeId)
			{
				$total++;
				$total += $this->importNodeTree($nodes, $tree, $oldNodeId);
			}
		}

		return $total;
	}

	// ############################## STEP: MODERATORS #############################

	public function stepModerators(StepState $state, array $stepConfig)
	{
		$this->typeMap('node');

		$moderators = $this->sourceDb->fetchAll("
			SELECT mods.*
			FROM moderator_cache AS mods
			INNER JOIN users AS users ON (mods.user_id = users.user_id)
		");
		if (!$moderators)
		{
			return $state->complete();
		}

		$modsGrouped = [];
		foreach ($moderators AS $moderator)
		{
			$modsGrouped[$moderator['user_id']][$moderator['forum_id']] = $moderator;
		}

		$this->lookup('user', array_keys($modsGrouped));

		/** @var Moderator $modHelper */
		$modHelper = $this->getDataHelper(Moderator::class);

		foreach ($modsGrouped AS $userId => $forums)
		{
			$newUserId = $this->lookupId('user', $userId);
			if (!$newUserId)
			{
				continue;
			}

			$inserted = false;

			foreach ($forums AS $forumId => $moderator)
			{
				$newNodeId = $this->lookupId('node', $forumId);
				if (!$newNodeId)
				{
					continue;
				}

				$permissions = $this->convertForumPermissionsForUser($moderator);
				$modHelper->importContentModerator($newUserId, 'node', $newNodeId, $permissions);

				$inserted = true;
			}

			if ($inserted)
			{
				$modHelper->importModerator($newUserId, false, [User::GROUP_MOD]);

				$state->imported++;
			}
		}

		return $state->complete();
	}

	// ############################## STEP: THREADS #############################

	public function getStepEndThreads()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(topic_id)
			FROM topics
		") ?: 0;
	}

	public function stepThreads(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$threads = $this->sourceDb->fetchAllKeyed("
			SELECT topics.*, topic_posts_approved AS topic_replies, topic_visibility AS topic_approved,
				IF(users.username IS NOT NULL, users.username, topics.topic_first_poster_name) AS username
			FROM topics AS topics FORCE INDEX (PRIMARY)
			LEFT JOIN users AS users ON (topics.topic_poster = users.user_id)
			INNER JOIN forums AS forums ON
				(topics.forum_id = forums.forum_id)
			WHERE topics.topic_id > ? AND topics.topic_id <= ?
				AND topics.topic_status <> 2
			ORDER BY topics.topic_id
			LIMIT {$limit}
		", 'topic_id', [$state->startAfter, $state->end]);

		if (!$threads)
		{
			return $state->complete();
		}

		$this->typeMap('node');
		$this->lookup('user', $this->pluck($threads, 'topic_poster'));

		foreach ($threads AS $oldThreadId => $thread)
		{
			$state->startAfter = $oldThreadId;

			$nodeId = $this->lookupId('node', $thread['forum_id']);

			if (!$nodeId)
			{
				continue;
			}

			/** @var Thread $import */
			$import = $this->newHandler(Thread::class);

			$import->bulkSet($this->mapXfKeys($thread, [
				'reply_count' => 'topic_replies',
				'view_count' => 'topic_views',
				'last_post_date' => 'topic_last_post_time',
				'post_date' => 'topic_time',
			]));

			$import->bulkSet([
				'discussion_open' => ($thread['topic_status'] == 0 ? 1 : 0),
				'node_id' => $nodeId,
				'user_id' => $this->lookupId('user', $thread['topic_poster'], 0),
				'discussion_state' => $thread['topic_approved'] ? 'visible' : 'moderated',
			]);

			$import->set('title', $thread['topic_title'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->set('username', $thread['username'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->set('last_post_username', $thread['topic_last_poster_name'], [EntityEmulator::UNHTML_ENTITIES => true]);

			$subs = $this->sourceDb->fetchPairs("
				SELECT user_id, notify_status
				FROM topics_watch
				WHERE topic_id = {$oldThreadId}
			");
			if ($subs)
			{
				$this->lookup('user', array_keys($subs));

				foreach ($subs AS $userId => $emailSubscribe)
				{
					$newUserId = $this->lookupId('user', $userId);
					if (!$newUserId)
					{
						continue;
					}

					$import->addThreadWatcher($newUserId, $emailSubscribe);
				}
			}

			if ($newThreadId = $import->save($oldThreadId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ############################## STEP: POSTS #############################

	protected function getThreadIdsForPostsStep($startAfter, $end, $threadLimit)
	{
		return $this->sourceDb->fetchAllColumn("
			SELECT topic_id
			FROM topics
			WHERE topic_id > ? AND topic_id <= ?
			ORDER BY topic_id
			LIMIT {$threadLimit}
		", [$startAfter, $end]);
	}

	protected function getPostsForPostsStep($threadId, $startDate)
	{
		$limit = self::$postsStepLimit;

		return $this->sourceDb->fetchAll("
			SELECT posts.*, post_visibility AS post_approved,
				IF(users.username IS NOT NULL, users.username, posts.post_username) AS username
			FROM posts AS posts
			LEFT JOIN users AS users ON (posts.poster_id = users.user_id)
			WHERE posts.topic_id = ?
				AND posts.post_time > ?
			ORDER BY posts.post_time
			LIMIT {$limit}
		", [$threadId, $startDate]);
	}

	protected function lookupUsers(array $posts)
	{
		$this->lookup('user', $this->pluck($posts, 'poster_id'));
	}

	protected function getPostDateField()
	{
		return 'post_time';
	}

	protected function getPostIdField()
	{
		return 'post_id';
	}

	protected function handlePostImport(array $post, $newThreadId, StepState $state)
	{
		/** @var Post $import */
		$import = $this->newHandler(Post::class);

		$import->bulkSet([
			'thread_id' => $newThreadId,
			'user_id' => $this->lookupId('user', $post['poster_id']),
			'post_date' => $post['post_time'],
			'message' => $this->convertContentToBbCode($post['post_text']),
			'message_state' => $post['post_approved'] ? 'visible' : 'moderated',
			'position' => $state->extra['postPosition'],
		]);

		$import->set('username', $post['username'], [EntityEmulator::UNHTML_ENTITIES => true]);

		$import->setLoggedIp($post['poster_ip']);

		return $import;
	}

	// ########################### STEP: POLLS ###############################

	public function getStepEndPolls()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(topic_id)
			FROM poll_options
		") ?: 0;
	}

	public function stepPolls(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$polls = $this->sourceDb->fetchAllKeyed("
			SELECT DISTINCT(polls.topic_id),
				topics.poll_title, topics.poll_start, topics.poll_length,
				topics.poll_max_options, topics.poll_last_vote, topics.poll_vote_change
			FROM poll_options AS polls
			INNER JOIN topics AS topics ON (topics.topic_id = polls.topic_id)
			WHERE polls.topic_id > ? AND polls.topic_id <= ?
			ORDER BY polls.topic_id
			LIMIT {$limit}
		", 'topic_id', [$state->startAfter, $state->end]);

		if (!$polls)
		{
			return $state->complete();
		}

		$pollsCompleted = [];

		$this->lookup('thread', $this->pluck($polls, 'topic_id'));

		foreach ($polls AS $oldId => $poll)
		{
			$state->startAfter = $oldId;

			$newThreadId = $this->lookupId('thread', $poll['topic_id']);
			if (!$newThreadId)
			{
				continue;
			}

			if (array_key_exists($oldId, $pollsCompleted))
			{
				// poll id in the thread table isn't unique, so use this to avoid duplication
				continue;
			}

			$pollsCompleted[$oldId] = true;

			/** @var Poll $import */
			$import = $this->newHandler(Poll::class);
			$import->bulkSet([
				'content_type' => 'thread',
				'content_id' => $newThreadId,
				'question' => $this->convertContentToBbCode($poll['poll_title'], true),
				'max_votes' => $poll['poll_max_options'],
				'close_date' => ($poll['poll_length'] ? $poll['poll_start'] + $poll['poll_length'] : 0),
			]);

			$responses = $this->sourceDb->fetchPairs('
				SELECT poll_option_id, poll_option_text
				FROM poll_options
				WHERE topic_id = ?
			', $oldId);
			foreach ($responses AS &$value)
			{
				$value = $this->convertContentToBbCode($value, true);
			}

			if (!$responses)
			{
				continue;
			}

			$importResponses = [];

			foreach ($responses AS $i => $responseText)
			{
				/** @var PollResponse $importResponse */
				$importResponse = $this->newHandler(PollResponse::class);
				$importResponse->preventRetainIds();
				$importResponse->response = $responseText;

				$importResponses[$i] = $importResponse;

				$import->addResponse($i, $importResponse);
			}

			$votes = $this->sourceDb->fetchAll("
				SELECT vote_user_id, poll_option_id
				FROM poll_votes
				WHERE topic_id = ?
			", $oldId);

			$this->lookup('user', $this->pluck($votes, 'vote_user_id'));

			foreach ($votes AS $vote)
			{
				$voteUserId = $this->lookupId('user', $vote['vote_user_id']);
				if (!$voteUserId)
				{
					continue;
				}

				$voteOption = $vote['poll_option_id'];

				if (!array_key_exists($voteOption, $importResponses))
				{
					continue;
				}

				$importResponses[$voteOption]->addVote($voteUserId);
			}

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: ATTACHMENTS ###############################

	public function getStepEndAttachments()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(attach_id)
			FROM attachments
		") ?: 0;
	}

	public function stepAttachments(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$attachments = $this->sourceDb->fetchAll("
			SELECT *
			FROM attachments
			WHERE attach_id > ? AND attach_id <= ?
				AND is_orphan = 0
				AND post_msg_id > 0
				AND in_message = 0
			ORDER BY attach_id
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$attachments)
		{
			return $state->complete();
		}

		$this->lookup('user', $this->pluck($attachments, 'poster_id'));
		$this->lookup('post', $this->pluck($attachments, 'post_msg_id'));

		$attachPath = $this->baseConfig['attach_path'];

		foreach ($attachments AS $attachment)
		{
			$state->startAfter = $attachment['attach_id'];

			$newPostId = $this->lookupId('post', $attachment['post_msg_id']);
			if (!$newPostId)
			{
				continue;
			}

			$sourceFile = $attachPath . \XF::$DS . $attachment['physical_filename'];
			if (!file_exists($sourceFile))
			{
				continue;
			}

			/** @var \XF\Import\Data\Attachment $import */
			$import = $this->newHandler(\XF\Import\Data\Attachment::class);
			$import->bulkSet([
				'content_type' => 'post',
				'content_id'   => $newPostId,
				'attach_date'  => $attachment['filetime'],
				'view_count'   => $attachment['download_count'],
				'unassociated' => false,
			]);
			$import->setDataExtra('upload_date', $attachment['filetime']);

			$import->setDataUserId($this->lookupId('user', $attachment['poster_id']));
			$import->setSourceFile($sourceFile, $this->convertToUtf8($attachment['real_filename'], null, true));
			$import->setContainerCallback([$this, 'rewriteEmbeddedAttachments']);

			$newId = $import->save($attachment['attach_id']);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		File::cleanUpTempFiles();

		return $state->resumeIfNeeded();
	}

	protected function convertForumPermissionsForUser(array $moderator)
	{
		$userId = $moderator['user_id'];
		$forumId = $moderator['forum_id'];

		$oldPerms = $this->getUserPermissions($userId, $forumId);
		$perms = ['forum' => []];

		if ($this->hasOldPermission($oldPerms, 'm_merge') || $this->hasOldPermission($oldPerms, 'm_move') || $this->hasOldPermission($oldPerms, 'm_split'))
		{
			$perms['forum']['manageAnyThread'] = 'content_allow';
			$perms['forum']['stickUnstickThread'] = 'content_allow';
		}
		if ($this->hasOldPermission($oldPerms, 'm_approve'))
		{
			$perms['forum']['approveUnapprove'] = 'content_allow';
			$perms['forum']['viewModerated'] = 'content_allow';
		}
		if ($this->hasOldPermission($oldPerms, 'm_delete'))
		{
			$perms['forum']['deleteAnyPost'] = 'content_allow';
			$perms['forum']['deleteAnyThread'] = 'content_allow';
			$perms['forum']['undelete'] = 'content_allow';
			$perms['forum']['viewDeleted'] = 'content_allow';
		}
		$this->convertOldPermissionValue($perms, $oldPerms, 'forum', 'editAnyPost', 'm_edit', 'content_allow');
		$this->convertOldPermissionValue($perms, $oldPerms, 'forum', 'lockUnlockThread', 'm_lock', 'content_allow');

		return $perms;
	}

	protected function convertContentToBbCode($string, $strip = false)
	{
		if ($this->isLegacyContent($string))
		{
			// how smilies are stored
			// note seems they are sometimes stored with double quoted attributes
			$string = preg_replace(
				'#<img src=""?[^"]*"?" alt=""?([^"]*)"?" title=""?[^"]*"?"\s*/?>#iU',
				'$1',
				$string
			);

			// seen some email links like this
			$string = preg_replace(
				'#<a[^>]+href="mailto:([^"]+)"[^>]*>(.*)</a>#siU',
				'[email=$1]$2[/email]',
				$string
			);

			// seen some links like this
			$string = preg_replace(
				'#<a[^>]+href="([^"]+)"[^>]*>(.*)</a>#siU',
				'[url=$1]$2[/url]',
				$string
			);

			// other comment markup
			$string = preg_replace('#<!--.*-->#siU', '', $string);

			$string = $this->convertToUtf8($string, null, true);

			do
			{
				$previousString = $string;

				// Handles quotes with arguments
				$string = preg_replace(
					'#\[quote=(?:"|)(.+)(?:"|)(?:\s?(?:post_id|time|user_id)=[^\]]+|)\]#siU',
					'[quote="$1"]',
					$string
				);

				$string = preg_replace(
					'#\[attachment=(\d+):([^]]+)\](.*)\[/attachment:\2\]#siU',
					'[ATTACH type="full" alt="$3"]$1[/ATTACH]',
					$string
				);

				// size tags need mapping
				$string = preg_replace_callback(
					'#\[(size)="?([^\]]*)"?:([a-z0-9]+)\](.*)\[/size:\\3\]#siU',
					[$this, 'convertBbCodeSize'],
					$string
				);

				// align tags need mapping
				$string = preg_replace(
					'#\[align="?(left|center|right)"?:([a-z0-9]+)\](.*)\[/align:\\2\]#siU',
					'[$1]$3[/$1]',
					$string
				);
			}
			while ($string != $previousString);

			$string = preg_replace(
				'#\[([a-z0-9_\*]+(="[^"]*"|=[^\]]*)?):([a-z0-9]+)\]#siU',
				($strip ? '' : '[$1]'),
				$string
			);
			$string = preg_replace(
				'#\[/([a-z0-9_\*]+)(:[a-z])?:([a-z0-9]+)\]#siU',
				($strip ? '' : '[/$1]'),
				$string
			);

			$string = str_replace('[/*]', '', $string);

			return $string;
		}
		else
		{
			$string = html_entity_decode(strip_tags($string), ENT_QUOTES, 'UTF-8');

			$string = preg_replace('#<!--.*-->#siU', '', $string);
			$string = str_replace('[/*]', '', $string);
			$string = $this->convertToUtf8($string, null, true);

			do
			{
				$previousString = $string;

				// Handles quotes with arguments
				$string = preg_replace(
					'#\[quote=(?:"|)(.+)(?:"|)(?:\s?(?:post_id|time|user_id)=[^\]]+|)\]#siU',
					'[quote="$1"]',
					$string
				);

				$string = preg_replace(
					'#\[attachment=(\d+)\](.*)\[/attachment\]#siU',
					'[ATTACH type="full" alt="$2"]$1[/ATTACH]',
					$string
				);

				// size tags need mapping
				$string = preg_replace_callback(
					'#\[(size)="?([^\]]*)"?\](.*)\[/size\]#siU',
					[$this, 'convertBbCodeSize'],
					$string
				);

				// align tags need mapping
				$string = preg_replace(
					'#\[align="?(left|center|right)"?\](.*)\[/align\]#siU',
					'[$1]$3[/$1]',
					$string
				);
			}
			while ($string != $previousString);

			return $string;
		}
	}

	protected function isLegacyContent($string)
	{
		$start = substr($string, 0, 3);
		return ($start != '<r>' && $start != '<t>');
	}

	protected function convertBbCodeSize(array $match)
	{
		$tag = $match[1];
		$size = intval($match[2]);
		$text = $match[4] ?? $match[3];

		if ($size >= 200)
		{
			$size = 7;
		}
		else if ($size >= 170)
		{
			$size = 6;
		}
		else if ($size >= 140)
		{
			$size = 5;
		}
		else if ($size >= 110)
		{
			$size = 4;
		}
		else if ($size >= 90)
		{
			$size = 3;
		}
		else if ($size >= 60)
		{
			$size = 2;
		}
		else if ($size <= 7)
		{
			// keep size as is
		}
		else
		{
			$size = 1;
		}

		return '[' . $tag . '=' . $size . ']' . $text . '[/' . $tag . ']';
	}

	public function rewriteEmbeddedAttachments(Entity $container, Attachment $attachment, $oldId, array $extras, $messageCol = 'message')
	{
		if (isset($container->$messageCol))
		{
			$filename = preg_quote($attachment->getFilename(), '#');
			$message = $container->$messageCol;

			$message = preg_replace_callback(
				"#(\[ATTACH type=\"full\" alt=\"{$filename}\"])\d+(\[\/ATTACH])#siU",
				function ($match) use ($attachment, $container)
				{
					$id = $attachment->attachment_id;

					if (isset($container->embed_metadata))
					{
						$metadata = $container->embed_metadata;
						$metadata['attachments'][$id] = $id;

						$container->embed_metadata = $metadata;
					}

					/*
					 * Note: We use '$id._xfImport' as the attachment id in the XenForo replacement
					 * to avoid it being replaced again if we come across an attachment whose source id
					 * is the same as this one's imported id.
					 */

					return $match[1] . $id . '._xfImport' . $match[2];
				},
				$message,
				1
			);

			$container->$messageCol = $message;
		}
	}
}
