<?php

namespace XFI\Import\Importer;

use XF\Db\AbstractAdapter;
use XF\Entity\FeaturedContent;
use XF\Entity\Thread;
use XF\Import\StepState;
use XF\Job\ConversationEmbedMetadata;
use XF\Job\PostEmbedMetadata;
use XF\Job\ProfilePostCommentEmbedMetadata;
use XF\Job\ProfilePostEmbedMetadata;

use function in_array, intval;

class XenForo23 extends XenForo22
{
	public static function getListInfo(): array
	{
		return [
			'target' => 'XenForo',
			'source' => 'XenForo 2.3',
		];
	}

	/**
	 * @param string|\Stringable|null &$error
	 */
	protected function validateVersion(AbstractAdapter $db, &$error): bool
	{
		$versionId = $db->fetchOne(
			'SELECT option_value
				FROM xf_option
				WHERE option_id = ?',
			['currentVersionId']
		);
		if (!$versionId || intval($versionId) < 2030031)
		{
			$error = \XF::phrase('xfi_you_may_only_import_from_xenforo_x', [
				'version' => '2.3',
			]);
			return false;
		}

		return true;
	}

	public function getSteps(): array
	{
		$steps = parent::getSteps();

		$steps = $this->extendSteps(
			$steps,
			[
				'title' => \XF::phrase('featured_threads'),
				'depends' => ['threads'],
			],
			'featuredThreads',
			'threadPolls'
		);

		return $steps;
	}

	public function getStepEndFeaturedThreads(): int
	{
		return $this->getMaxFeaturedContentIdForContentType('thread');
	}

	public function stepFeaturedThreads(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		return $this->getFeatureStepStateForContentType(
			'thread',
			$state,
			$stepConfig,
			$maxTime,
			function (Thread $content, FeaturedContent $feature): void
			{
				$content->fastUpdate('featured', true);
			}
		);
	}

	protected function rewriteMessage(
		string $text,
		string $importType = 'post'
	): string
	{
		$text = parent::rewriteMessage($text, $importType);

		$text = $this->rewriteEmbeds($text);

		return $text;
	}

	public function getFinalizeJobs(array $stepsRun): array
	{
		$jobs = parent::getFinalizeJobs($stepsRun);

		if (in_array('conversations', $stepsRun, true))
		{
			$jobs[] = [
				ConversationEmbedMetadata::class,
				['types' => ['embeds']],
			];
		}

		if (in_array('profilePosts', $stepsRun, true))
		{
			$jobs[] = [
				ProfilePostEmbedMetadata::class,
				['types' => ['embeds']],
			];
			$jobs[] = [
				ProfilePostCommentEmbedMetadata::class,
				['types' => ['embeds']],
			];
		}

		if (in_array('posts', $stepsRun, true))
		{
			$jobs[] = [
				PostEmbedMetadata::class,
				['types' => ['embeds']],
			];
		}

		return $jobs;
	}
}
