<?php

namespace XFI\Import\Importer;

use XF\Entity\Attachment;
use XF\Import\Data\EntityEmulator;
use XF\Import\Data\Forum;
use XF\Import\DataHelper\Moderator;
use XF\Import\DataHelper\Permission;
use XF\Import\DataHelper\Tag;
use XF\Import\StepState;
use XF\Mvc\Entity\Entity;
use XF\Repository\Node;
use XF\Repository\NodeRepository;
use XF\Timer;
use XF\Util\File;
use XFI\Import\Data\vBulletinBlogAttachment;
use XFI\Import\Data\vBulletinBlogEntry;
use XFI\Import\Data\vBulletinBlogText;
use XFI\Import\Data\vBulletinBlogUser;

trait vBulletinBlogTrait
{
	public function getSteps()
	{
		return parent::getSteps() + [
			//				'blogUsers' => [
			//					'title' => \XF::phrase('xfi_import_blogs'),
			//					'depends' => ['users']
			//				],
			'blogEntries' => [
				'title' => \XF::phrase('xfi_import_blog_entries'),
				'depends' => ['users'],
			],
			'blogAttachments' => [
				'title' => \XF::phrase('xfi_import_blog_attachments'),
				'depends' => ['blogEntries'],
			],
			'blogComments' => [
				'title' => \XF::phrase('xfi_import_blog_comments'),
				'depends' => ['blogEntries'],
			],
			'blogTags' => [
				'title' => \XF::phrase('xfi_import_blog_tags'),
				'depends' => ['blogEntries'],
			],
			'blogModerators' => [
				'title' => \XF::phrase('xfi_import_blog_moderators'),
				'depends' => ['blogUsers'],
			],
		];
	}

	protected function getStepConfigDefault()
	{
		return parent::getStepConfigDefault() + [
			/*'_blogUsers' => [
					'parent_node_id' => 0,
					'parent_node_title' => \XF::phrase('xfi_import_blogs')
				],*/
			'blogAttachments' => [
				'path' => $this->baseConfig['blogattachpath'],
			],
			'blogAttachmentsVB4' => [
				'path' => $this->baseConfig['attachpath'],
			],
		];
	}

	protected function getStepConfigOptions(array $vars)
	{
		$vars = parent::getStepConfigOptions($vars);

		if (in_array('blogUsers', $vars['steps']) || in_array('blogEntries', $vars['steps']))
		{
			if (!isset($vars['nodeTree']))
			{
				/** @var Node $nodeRepo */
				$nodeRepo = $this->app->repository(NodeRepository::class);
				$vars['nodeTree'] = $nodeRepo->createNodeTree($nodeRepo->getFullNodeList());
			}
		}

		return $vars;
	}

	/**
	 * Ensure that there is a valid title set for Blogs parent, when $retainIds is set
	 *
	 * @param array $config
	 * @param array $stepConfig
	 * @param array $errors
	 */
	protected function validateStepConfigBlogUsers(array &$config, array &$stepConfig, array &$errors)
	{
		if (!empty($stepConfig['_retainIds']))
		{
			if (trim($config['parent_node_title']) === '')
			{
				$errors['blogUsers'] = \XF::phrase('please_enter_valid_title');
			}
		}
	}

	/**
	 * Ensure that blogattachpath is valid
	 *
	 * @param array $config
	 * @param array $stepConfig
	 * @param array $errors
	 */
	protected function validateStepConfigBlogAttachments(array &$config, array &$stepConfig, array &$errors)
	{
		// blog attachments as files - path
		if (!empty($this->baseConfig['blogattachpath']))
		{
			if (!empty($config['path']))
			{
				$path = realpath(trim($config['path']));

				if (!file_exists($path) || !is_dir($path))
				{
					$errors[] = \XF::phrase('directory_specified_as_x_y_not_found_is_not_readable', [
						'type' => 'blogattachpath',
						'dir'  => $config['path'],
					]);
				}

				$config['path'] = $path;
			}
			else
			{
				// we have no path, this will cause the step to skip
				$config['skip'] = true;
			}
		}
	}

	// ########################### STEP: BLOG USERS ###############################

	/**
	 * @deprecated
	 */
	public function _getStepEndBlogUsers()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(bloguserid)
			FROM blog_user
			WHERE entries > 0") ?: 0;
	}

	/**
	 * @deprecated
	 */
	public function _stepBlogUsers(StepState $state, array $stepConfig, $maxTime, $limit = 100)
	{
		$timer = new Timer($maxTime);

		$blogUsers = $this->sourceDb->fetchAllKeyed("
			SELECT
				bu.bloguserid,
				IF(bu.title <> '', bu.title, user.username) AS title,
				bu.description,
				user.username
			FROM blog_user AS
				bu
			INNER JOIN user AS
				user ON (user.userid = bu.bloguserid)
			WHERE bu.bloguserid > ? AND bu.bloguserid <= ?
			AND bu.entries > 0
			ORDER BY bu.bloguserid
		", 'bloguserid', [$state->startAfter, $state->end]);

		if (!$blogUsers)
		{
			return $state->complete();
		}

		$blogsParentNodeId = $this->getBlogsParentNodeId($stepConfig);

		$subs = $this->sourceDb->fetchAll("
			SELECT bloguserid, userid, type
			FROM blog_subscribeuser
			WHERE bloguserid IN(" . $this->sourceDb->quote(array_keys($blogUsers)) . ")
		");
		$blogUserSubscriptions = [];
		foreach ($subs AS $sub)
		{
			$blogUserSubscriptions[$sub['bloguserid']][$sub['userid']] = ($sub['type'] == 'email');
		}

		$this->lookup('user', array_keys($blogUsers) + $this->pluck($subs, 'userid'));

		foreach ($blogUsers AS $oldUserId => $blogUser)
		{
			$state->startAfter = $oldUserId;

			if (!$newUserId = $this->lookupId('user', $oldUserId))
			{
				continue;
			}

			/** @var vBulletinBlogUser $import */
			$import = $this->newHandler(vBulletinBlogUser::class);
			$import->preventRetainIds();

			$import->title = $blogUser['title'];
			$import->description = $blogUser['description'];
			$import->parent_node_id = $blogsParentNodeId;

			/** @var Forum $forumHandler */
			$forumHandler = $this->newHandler(Forum::class);

			if (isset($blogUserSubscriptions[$oldUserId]))
			{
				foreach ($blogUserSubscriptions[$oldUserId] AS $oldSubUserId => $emailUpdate)
				{
					if ($newSubUserId = $this->lookupId('user', $oldSubUserId))
					{
						$forumHandler->addForumWatcher($newSubUserId, $this->setupForumSubscribeData($emailUpdate));
					}
				}
			}

			$import->setType('Forum', $forumHandler);

			if ($newId = $import->save($oldUserId))
			{
				/** @var Permission $permissionHelper */
				$permissionHelper = $this->getDataHelper(Permission::class);

				// deny forum.newThread permissions to registered users, and allow for the blog owner
				$permissionHelper->insertContentUserGroupPermissions('node', $newId, 2, ['forum' => ['postThread' => 'reset']]);
				$permissionHelper->insertContentUserPermissions('node', $newId, $newUserId, ['forum' => ['postThread' => 'content_allow']]);

				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				continue;
			}
		}

		return $state->resumeIfNeeded();
	}

	/**
	 * When $retainIds is set, creates a forum node in order to store all
	 * imported blogs.
	 *
	 * This should not be called within a step, as import parallelization will
	 * cause multiple nodes to be created. This should be called during step
	 * setup instead. This has no effect if the blog node parent has already
	 * been created.
	 *
	 * @param array $stepConfigBlogUsers
	 */
	protected function createBlogsParentNodeIfNecessary(array $stepConfigBlogUsers)
	{
		if (
			$this->session->retainIds &&
			empty($this->session->extra['blogUsersParentNodeId'])
		)
		{
			$this->createImportNodeParentIfNecessary();

			/** @var \XF\Import\Data\Node $parentNode */
			$parentNode = $this->newHandler(\XF\Import\Data\Node::class);

			$parentNode->parent_node_id = $this->getImportNodeParentId();
			$parentNode->title = $stepConfigBlogUsers['parent_node_title'];
			$parentNode->display_order = 10;

			$parentType = $this->newHandler(Forum::class);
			$parentNode->setType('Forum', $parentType);

			$this->session->extra['blogUsersParentNodeId'] = $parentNode->save(false);
		}
	}

	/**
	 * When $retainIds is set, returns the ID of the node created during step
	 * setup, or throws an exception if it has not been created.
	 *
	 * When $retainIds is not set, returns the ID of the chosen parent node.
	 *
	 * @param array $stepConfigBlogUsers
	 *
	 * @return int
	 * @throws \LogicException
	 */
	protected function getBlogsParentNodeId(array $stepConfigBlogUsers)
	{
		if ($this->session->retainIds)
		{
			if (empty($this->session->extra['blogUsersParentNodeId']))
			{
				throw new \LogicException('The blog node parent has not been created.');
			}

			return $this->session->extra['blogUsersParentNodeId'];
		}
		else
		{
			return $stepConfigBlogUsers['parent_node_id'];
		}
	}

	// ########################### STEP: BLOG ENTRIES ###############################

	public function setupStepBlogEntries(array $stepConfig)
	{
		$stepConfigBlogUsers = $this->getSession()->stepConfig['blogUsers'];
		$this->createBlogsParentNodeIfNecessary($stepConfigBlogUsers);
	}

	public function getStepEndBlogEntries()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(blogid) FROM blog") ?: 0;
	}

	public function stepBlogEntries(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$blogEntries = $this->sourceDb->fetchAllKeyed("
			SELECT
				blog.*, blog_text.*,
				user.username, blog.state AS state
			FROM blog AS
				blog
			INNER JOIN blog_text AS
				blog_text ON (blog_text.blogtextid = blog.firstblogtextid)
			INNER JOIN user AS
				user ON (user.userid = blog.userid)
			WHERE blog.blogid > ? AND blog.blogid <= ?
			ORDER BY blog.blogid
			LIMIT {$limit}
		", 'blogid', [$state->startAfter, $state->end]);

		if (!$blogEntries)
		{
			return $state->complete();
		}

		$blogUserStepConfig = $this->getSession()->stepConfig['blogUsers'];
		$newNodeId = $this->getBlogsParentNodeId($blogUserStepConfig);

		$subs = $this->sourceDb->fetchAll("
			SELECT blogid, userid, type
			FROM blog_subscribeentry
			WHERE blogid IN(" . $this->sourceDb->quote(array_keys($blogEntries)) . ")
		");
		$blogSubscriptions = [];
		foreach ($subs AS $sub)
		{
			$blogSubscriptions[$sub['blogid']][$sub['userid']] = ($sub['type'] == 'email');
		}

		$this->lookup('user', $this->pluck($blogEntries, 'userid') + $this->pluck($subs, 'userid'));

		foreach ($blogEntries AS $blogId => $blog)
		{
			$state->startAfter = $blogId;

			if (!$newUserId = $this->lookupId('user', $blog['userid']))
			{
				continue;
			}

			/** @var vBulletinBlogEntry $importThread */
			$importThread = $this->newHandler(vBulletinBlogEntry::class);
			$importThread->preventRetainIds();

			$importThread->node_id = $newNodeId;
			$importThread->user_id = $newUserId;

			$importThread->set('username', $blog['username'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$importThread->set('title', $blog['title'], [EntityEmulator::UNHTML_ENTITIES => true]);

			$importThread->bulkSet($this->mapXfKeys($blog, [
				'view_count' => 'views',
				'post_date' => 'dateline',
				'last_post_date' => 'lastcomment',
			]));

			$importThread->discussion_open = true; // TODO: not sure this is totally accurate..?
			$importThread->discussion_state = $this->decodeBlogState($blog['state']);

			$blogText = $this->processBlogText($blog['pagetext']);

			$importThread->setBlogText($blogText, $blog['blogtextid']);
			$importThread->setLoggedIp($this->getLoggedIp($blog['ipaddress']));

			if (isset($blogSubscriptions[$blogId]))
			{
				foreach ($blogSubscriptions[$blogId] AS $oldSubUserId => $emailUpdate)
				{
					if ($newSubUserId = $this->lookupId('user', $oldSubUserId))
					{
						$importThread->addThreadWatcher($newSubUserId, $emailUpdate);
					}
				}
			}

			if ($newThreadId = $importThread->save($blogId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected function decodeBlogState($state)
	{
		switch ($state)
		{
			case 'moderation':
				return 'moderated';
			case 'draft':
				return 'visible'; // TODO: a draft state?
			case 'deleted':
			case 'visible':
				return $state;
		}
	}

	// ########################### STEP: BLOG ATTACHMENTS (vB3-style) ###############################

	public function getStepEndBlogAttachments()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(attachmentid) FROM blog_attachment") ?: 0;
	}

	public function stepBlogAttachments(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		if (isset($stepConfig['skip']))
		{
			return $state->complete();
		}

		if (!$state->startAfter && !$state->end)
		{
			if (!$this->blogAttachmentTableExists())
			{
				return $state->complete();
			}
		}

		$timer = new Timer($maxTime);

		$attachments = $this->sourceDb->fetchAllKeyed("
			SELECT
				atc.attachmentid, atc.blogid, atc.userid,
				atc.filename, atc.filesize, atc.visible,
				atc.counter, atc.dateline,
				atc.extension,
				blog.firstblogtextid AS blogtextid
			FROM blog_attachment AS
				atc
			INNER JOIN blog AS
				blog ON (blog.blogid = atc.blogid)
			WHERE atc.attachmentid > ? AND atc.attachmentid <= ?
			LIMIT {$limit}
		", 'attachmentid', [$state->startAfter, $state->end]);

		if (!$attachments)
		{
			return $state->complete();
		}

		$this->lookup('blog_text', $this->pluck($attachments, 'blogtextid'));
		$this->lookup('user', $this->pluck($attachments, 'userid'));

		foreach ($attachments AS $attachmentId => $attachment)
		{
			$state->startAfter = $attachmentId;

			if (!$newBlogTextId = $this->lookupId('blog_text', $attachment['blogtextid']))
			{
				continue;
			}

			if ($stepConfig['path'])
			{
				// original attachment file location
				$attachTempFile =  $this->getAttachmentFilePathVB3($stepConfig['path'], $attachment);

				if (!file_exists($attachTempFile))
				{
					continue;
				}
			}
			else
			{
				$fileData = $this->sourceDb->fetchOne("
					SELECT filedata
					FROM blog_attachment
					WHERE attachmentid = ?
				", $attachmentId);

				if (!$fileData)
				{
					continue;
				}

				$attachTempFile = File::getTempFile();
				File::writeFile($attachTempFile, $fileData);
			}

			/** @var vBulletinBlogAttachment $import */
			$import = $this->newHandler(vBulletinBlogAttachment::class);
			$import->preventRetainIds();

			$import->bulkSet([
				'content_type' => 'post',
				'content_id' => $newBlogTextId,
				'attach_date' => $attachment['dateline'],
				'view_count' => $attachment['counter'],
				'unassociated' => false,
			]);

			$import->setDataUserId($this->lookupId('user', $attachment['userid'], 0));
			$import->setSourceFile($attachTempFile, $attachment['filename']);
			$import->setContainerCallback([$this, 'rewriteEmbeddedBlogAttachments']);

			if ($newId = $import->save($attachmentId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		File::cleanUpTempFiles();

		return $state->resumeIfNeeded();
	}

	protected function blogAttachmentTableExists()
	{
		// The vB3 blog_attachment table may not exist,
		// in the case where this installation started with vB4
		// instead of being upgraded from vB3
		$prefix = $this->baseConfig['db']['tablePrefix'];
		return $this->sourceDb->fetchOne("SHOW TABLES LIKE '{$prefix}blog_attachment'");
	}

	public function rewriteEmbeddedBlogAttachments(Entity $container, Attachment $attachment, $oldId, $extras = [], $messageCol = 'message')
	{
		$this->rewriteEmbeddedAttachments($container, $attachment, $oldId, $extras, $messageCol);
	}

	// ########################### STEP: BLOG COMMENTS ###############################

	public function getStepEndBlogComments()
	{
		/*
		 * This assumes that blogs with comments will have a 'lastcomment' date later than the blog's own 'dateline'
		 */
		return $this->sourceDb->fetchOne("
			SELECT MAX(blogid)
			FROM blog
			WHERE lastcomment > dateline
		") ?: 0;
	}

	public function stepBlogComments(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$blogs = $this->sourceDb->fetchPairs("
			SELECT blogid, firstblogtextid
			FROM blog
			WHERE blogid > ? AND blogid <= ?
			AND lastcomment > dateline
			ORDER BY blogid
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		$this->lookup('blog', array_keys($blogs));

		foreach ($blogs AS $blogId => $firstBlogTextId)
		{
			$state->startAfter = $blogId;

			/*
			 * We will attempt to import ALL comments for each blog in one hit,
			 * on the assumption that we're unlikely to have thousands of comments
			 * in most cases...
			 */
			$blogComments = $this->sourceDb->fetchAll("
				SELECT
					blog_text.*,
					user.username,
					blog_editlog.userid AS edituserid,
					blog_editlog.dateline AS editdate,
					blog_editlog.reason AS editreason
				FROM blog_text AS
					blog_text
				LEFT JOIN blog_editlog AS
					blog_editlog ON (blog_editlog.blogtextid = blog_text.blogtextid)
				LEFT JOIN user AS
					user ON (user.userid = blog_text.userid)
				WHERE blog_text.blogid = ?
				AND blog_text.blogtextid <> ?
				ORDER BY blog_text.dateline
			", [$blogId, $firstBlogTextId]);

			if (!$blogComments)
			{
				continue;
			}

			$this->lookup('user', $this->pluck($blogComments, ['userid', 'edituserid']));

			$postPosition = 0;

			foreach ($blogComments AS $comment)
			{
				if (!$newThreadId = $this->lookupId('blog', $blogId))
				{
					continue;
				}

				$message = $this->processBlogCommentText($comment['pagetext']);

				if ($comment['state'] == 'visible')
				{
					$postPosition++;
				}

				/** @var vBulletinBlogText $import */
				$import = $this->newHandler(vBulletinBlogText::class);
				$import->preventRetainIds();

				$import->set('username', $comment['username'], [EntityEmulator::UNHTML_ENTITIES => true]);
				$import->bulkSet([
					'thread_id' => $newThreadId,
					'post_date' => $comment['dateline'],
					'user_id' => $this->lookupId('user', $comment['userid'], 0),
					'position' => $postPosition,
					'message' => $message,
					'message_state' => $this->decodeBlogState($comment['state']),
					'last_edit_date' => $comment['editdate'] ?: 0,
					'last_edit_user_id' => $this->lookupId('user', $comment['edituserid'], 0),
					'edit_count' => $comment['editdate'] ? 1 : 0,
				]);

				if ($newId = $import->save($comment['blogtextid']))
				{
					$state->imported++;
				}
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: BLOG TAGS ###############################

	public function getStepEndBlogTags()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(blogid) FROM blog
			WHERE taglist IS NOT NULL
		") ?: 0;
	}

	public function stepBlogTags(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$blogs = $this->getBlogIdsWithTags($state->startAfter, $state->end, $limit);

		if (!$blogs)
		{
			return $state->complete();
		}

		/** @var Tag $tagHelper */
		$tagHelper = $this->getDataHelper(Tag::class);

		foreach ($blogs AS $oldBlogId => $blogPostDate)
		{
			$state->startAfter = $oldBlogId;

			if (!$newThreadId = $this->lookupId('blog', $oldBlogId))
			{
				continue;
			}

			$tags = $this->getBlogTags($oldBlogId);

			if (!$tags)
			{
				continue;
			}

			$this->lookup('user', $this->pluck($tags, 'userid'));

			foreach ($tags AS $tag)
			{
				$newId = $tagHelper->importTag(htmlspecialchars_decode($tag['tagtext']), 'thread', $newThreadId, [
					'add_user_id' => $this->lookupId('user', $tag['userid']),
					'add_date' => $tag['dateline'],
					'visible' => 1,
					'content_date' => $blogPostDate,
				]);

				if ($newId)
				{
					$state->imported++;
				}
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected function getBlogIdsWithTags($startAfter, $end, $limit)
	{
		return $this->sourceDb->fetchPairs("
			SELECT blogid, dateline
			FROM blog
			WHERE blogid > ? AND blogid <= ?
			AND taglist IS NOT NULL
			ORDER BY blogid
			LIMIT {$limit}
		", [$startAfter, $end]);
	}

	protected function getBlogTags($blogId)
	{
		return $this->sourceDb->fetchAll("
			SELECT
				blog_tagentry.*, blog_tag.tagtext
			FROM blog_tagentry AS
				blog_tagentry
			INNER JOIN blog_tag AS
				blog_tag ON (blog_tag.tagid = blog_tagentry.tagid)
			WHERE blog_tagentry.blogid = ?
		", $blogId);
	}

	// ########################### STEP: BLOG MODERATORS ###############################

	public function setupStepBlogModerators(array $stepConfig)
	{
		$stepConfigBlogUsers = $this->getSession()->stepConfig['blogUsers'];
		$this->createBlogsParentNodeIfNecessary($stepConfigBlogUsers);
	}

	public function stepBlogModerators(StepState $state)
	{
		$moderators = $this->sourceDb->fetchAll("
			SELECT blog_moderator.*, user.username
			FROM blog_moderator AS
				blog_moderator
			INNER JOIN user AS
				user ON (user.userid = blog_moderator.userid)
		");

		if (!$moderators)
		{
			return $state->complete();
		}

		$this->lookup('users', $this->pluck($moderators, 'userid'));

		$blogsParentNodeId = $this->getBlogsParentNodeId($this->session->stepConfig['blogUsers']);

		/** @var Moderator $modHelper */
		$modHelper = $this->getDataHelper(Moderator::class);

		/** @var Permission $permissionHelper */
		$permissionHelper = $this->getDataHelper(Permission::class);

		// TODO: these might not be totally ideal mappings
		$forumMap = [
			0 => ['forum' => ['editAnyThread']],
			1 => ['forum' => ['deleteAnyThread']],
			2 => ['forum' => ['hardDeleteAnyThread']],
			3 => ['forum' => ['approveUnapprove', 'viewModerated']],
			4 => ['forum' => ['editAnyPost']],
			5 => ['forum' => ['deleteAnyPost']],
			6 => ['forum' => ['hardDeleteAnyPost']],
			7 => ['forum' => ['approveUnapprove', 'viewModerated']],
		];

		$generalMap = [
			8 => ['general' => ['viewIps']],
		];

		foreach ($moderators AS $moderator)
		{
			if (!$newUserId = $this->lookupId('user', $moderator['userid']))
			{
				continue;
			}

			$globalPermissions = $this->applyModeratorPermissionMap($moderator['permissions'], $generalMap, 'allow');

			$forumPermissions = $this->applyModeratorPermissionMap($moderator['permissions'], $forumMap, 'content_allow');

			$this->addDefaultModeratorPermissions($globalPermissions, $forumPermissions);

			$permissionHelper->insertUserPermissions($newUserId, $globalPermissions);
			$modHelper->importContentModerator($newUserId, 'node', $blogsParentNodeId, $forumPermissions);

			// TODO: what are 'super' blog moderators? (blog_moderator.type=normal|super)
			$modHelper->importModerator($newUserId, false, [4]);

			$state->imported++;
		}

		return $state->complete();
	}

	// ########################### TEXT PROCESSING ###############################

	protected function processBlogText($text)
	{
		return $this->rewriteQuotes(
			$this->cleanUpRte($text),
			$this->getBlogTextIdQuotePattern()
		);
	}

	protected function processBlogCommentText($text)
	{
		return $this->rewriteQuotes(
			$this->cleanUpRte($text),
			$this->getBlogTextIdQuotePattern()
		);
	}

	protected function getBlogTextIdQuotePattern()
	{
		return [
			'/\[QUOTE=((?P<quoted>\'|")?)(?P<username>(?(quoted)[^;\n]+|[^\][;\n]+))(;\s*(?P<blogtextid>\d+))\1\]/siU' => function ($match)
			{
				return sprintf(
					'[QUOTE="%s, post: %d"]',
					$match['username'],
					$this->lookupId('blog_text', $match['blogtextid'])
				);
			},
		];
	}
}
